/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2020-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GPUChemistryModel.H"
#include "reactingMixture.H"
#include "UniformField.H"
#include "extrapolatedCalculatedFvPatchgpuFields.H"
#include "IFstream.H"
#include "opencc.h"
#include "OFstream.H"
#include <sys/time.h>
struct my_timer
{
    struct timeval start_time, end_time;
    double time_use;

    void start()
    {
        gettimeofday(&start_time, NULL);
    }

    void stop()
    {
        gettimeofday(&end_time, NULL);
        time_use = (end_time.tv_sec - start_time.tv_sec) + (double)(end_time.tv_usec - start_time.tv_usec)/1000000.0;
    }
};


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class ReactionThermo, class ThermoType>
Foam::GPUChemistryModel<ReactionThermo, ThermoType>::GPUChemistryModel
(
    ReactionThermo& thermo
)
:
    BasicChemistryModel<ReactionThermo>(thermo),
    Y_(this->thermo().composition().Y()),
    reactions_
    (
        dynamic_cast<const reactingMixture<ThermoType>&>(this->thermo())
    ),
    specieThermo_
    (
        dynamic_cast<const reactingMixture<ThermoType>&>
            (this->thermo()).speciesData()
    ),
    nSpecie_(Y_.size()),
    nReaction_(reactions_.size()),
    Treact_
    (
        BasicChemistryModel<ReactionThermo>::template getOrDefault<scalar>
        (
            "Treact",
            0.0
        )
    ),
    RR_(nSpecie_)
{
    // Create the fields for the chemistry sources
    forAll(RR_, fieldi)
    {
        RR_.set
        (
            fieldi,
            new volScalargpuField::Internal
            (
                IOobject
                (
                    "RR." + Y_[fieldi].name(),
                    this->mesh().time().timeName(),
                    this->mesh().hostmesh(),
                    IOobject::NO_READ,
                    IOobject::NO_WRITE
                ),
                this->mesh(),
                dimensionedScalar(dimMass/dimVolume/dimTime, Zero)
            )
        );
    }

    IFstream tmp("./constant/thermophysicalProperties");

    dictionary thermoDict(tmp);

    IFstream tmp2(thermoDict.get<fileName>("foamChemistryFile").expand());

    dictionary chemDict_(tmp2);

    const speciesTable& species = this->thermo().composition().species();

    typedef typename Reaction<ThermoType>::specieCoeffs specieCoeffs;

    elements_const elements_const_ref;
    species_const  species_const_ref;
    thermo_ptr     thermo_ptr_ref;
    reactions_ptr  reactions_ptr_ref;

    // elements_const
    wordList elems;

    if (!chemDict_.readIfPresent("elements", elems))
    {
        Info<< "    elements not defined in " << chemDict_.name() << endl;
        return;
    }

    label ele_num = elements_const_ref.ele_num = elems.size();
    elements_const_ref.elements = (configItem*)malloc(sizeof(configItem));

    for (label i = 0; i < ele_num; ++i)
    {
        std::strcpy(elements_const_ref.elements[i].ele_names, elems[i].c_str());
        elements_const_ref.elements[i].ele_W = 1.0;
    }

    // species_const
    label sp_num = thermo_ptr_ref.sp_num = species_const_ref.sp_num = nSpecie_;

    species_const_ref.sp_W = (scalar *)malloc(sp_num*sizeof(scalar));
    species_const_ref.sp_name = (char *)malloc(100*sp_num*sizeof(char));
    species_const_ref.sp_nasa = (scalar *)malloc(14*sp_num*sizeof(scalar));
	species_const_ref.T_range = (scalar *)malloc(3*sp_num*sizeof(scalar));
  
    // thermo_ptr
    for (label i = 0; i < nSpecie_; ++i)
    {
        std::strcpy(species_const_ref.sp_name + 100*i, Y_[i].member().c_str());

        const label Index = species[Y_[i].member()];

        species_const_ref.sp_W[i] = specieThermo_[Index].W();

        species_const_ref.T_range[i*3 + 0] = specieThermo_[Index].Tlow();
        species_const_ref.T_range[i*3 + 1] = specieThermo_[Index].Tcommon();
        species_const_ref.T_range[i*3 + 2] = specieThermo_[Index].Thigh();

        species_const_ref.sp_nasa[i*14+ 0] = specieThermo_[Index].lowCpCoeffs()[0]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 1] = specieThermo_[Index].lowCpCoeffs()[1]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 2] = specieThermo_[Index].lowCpCoeffs()[2]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 3] = specieThermo_[Index].lowCpCoeffs()[3]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 4] = specieThermo_[Index].lowCpCoeffs()[4]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 5] = specieThermo_[Index].lowCpCoeffs()[5]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 6] = specieThermo_[Index].lowCpCoeffs()[6]/specieThermo_[Index].R();

        species_const_ref.sp_nasa[i*14+ 7] = specieThermo_[Index].highCpCoeffs()[0]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 8] = specieThermo_[Index].highCpCoeffs()[1]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 9] = specieThermo_[Index].highCpCoeffs()[2]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 10] = specieThermo_[Index].highCpCoeffs()[3]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 11] = specieThermo_[Index].highCpCoeffs()[4]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 12] = specieThermo_[Index].highCpCoeffs()[5]/specieThermo_[Index].R();
        species_const_ref.sp_nasa[i*14+ 13] = specieThermo_[Index].highCpCoeffs()[6]/specieThermo_[Index].R();
        
        
    }

    //reactions_ptr
    label react_num = reactions_ptr_ref.react_num = nReaction_;

    reactions_ptr_ref.tb_coeffs = (scalar *)malloc(sp_num*react_num*sizeof(scalar));
    reactions_ptr_ref.react_type = (scalar *)malloc(react_num*sizeof(scalar));
	reactions_ptr_ref.fall_type = (scalar *)malloc(react_num*sizeof(scalar));
    reactions_ptr_ref.fall_coeffs = (scalar *)malloc(5*react_num*sizeof(scalar));
    reactions_ptr_ref.vf = (scalar *)malloc(sp_num*react_num*sizeof(scalar));
    reactions_ptr_ref.vr = (scalar *)malloc(sp_num*react_num*sizeof(scalar));
    reactions_ptr_ref.is_rev = (scalar *)malloc(react_num*sizeof(scalar));
    reactions_ptr_ref.abe = (scalar *)malloc(6*react_num*sizeof(scalar));
    reactions_ptr_ref.v_net = (scalar *)malloc(sp_num*react_num*sizeof(scalar));
     

    dictionary& tmp_dict = chemDict_.subDict("reactions");

    for (label i=0; i < react_num; i++)
    {

        const dictionary& subDict_ = tmp_dict.subDict(reactions_[i].name());

        const word name = subDict_.get<word>("type");

        reactions_ptr_ref.react_type[i] = 0.;
        reactions_ptr_ref.fall_type[i] = 0.;

        for (label j = 0; j < sp_num; j++)
        {
            reactions_ptr_ref.tb_coeffs[j + sp_num*i] = 1.0;
        }

        if (name.find("thirdBody") != std::string::npos)
        {
            reactions_ptr_ref.react_type[i] = 1.;

            for (label j = 0; j < sp_num; j++)
            {
                auto coeffs = subDict_.get<List<Tuple2<word, scalar>>>("coeffs");
                reactions_ptr_ref.tb_coeffs[j + sp_num*i] = coeffs[j].second();
            }
            
        }
            
        
        if (name.find("FallOff") != std::string::npos)
        {

            reactions_ptr_ref.react_type[i] = 2.;
            reactions_ptr_ref.fall_type[i] = 1.;
            

            for (label j=0; j < sp_num; j++)
            {
                auto coeffs = subDict_.subDict("thirdBodyEfficiencies").get<List<Tuple2<word, scalar>>>("coeffs");
                reactions_ptr_ref.tb_coeffs[j + sp_num*i] = coeffs[j].second();
            }
            
            if (name.find("Troe") != std::string::npos)
            {
                reactions_ptr_ref.fall_type[i] = 2.;

                reactions_ptr_ref.fall_coeffs[5*i] = subDict_.subDict("F").get<scalar>("alpha");
                reactions_ptr_ref.fall_coeffs[5*i + 3] = subDict_.subDict("F").get<scalar>("Tsss");
                reactions_ptr_ref.fall_coeffs[5*i + 1] = subDict_.subDict("F").get<scalar>("Ts");
                reactions_ptr_ref.fall_coeffs[5*i + 2] = subDict_.subDict("F").get<scalar>("Tss");
            }

        } 

        if (name.find("irreversible") != std::string::npos)
        {
            reactions_ptr_ref.is_rev[i] = 0.;
        } 
        else
        {
            reactions_ptr_ref.is_rev[i] = 1.;
        }  

        const List<specieCoeffs>& lhs = reactions_[i].lhs();
        const List<specieCoeffs>& rhs = reactions_[i].rhs();

        for (label j = 0; j < sp_num; j++)
        {
            reactions_ptr_ref.vf[i + j*react_num] = 0.;
            reactions_ptr_ref.vr[i + j*react_num] = 0.;

            const label Index = species[Y_[j].member()];

            forAll(lhs, k)
            {
                const label specieI = lhs[k].index;
                
                if (Index == specieI)
                    reactions_ptr_ref.vf[i + j*react_num] += lhs[k].stoichCoeff;

            }

            forAll(rhs, k)
            {
                const label specieI = rhs[k].index;
                
                if (Index == specieI)
                    reactions_ptr_ref.vr[i + j*react_num] += rhs[k].stoichCoeff;

            }
            
        }


        for (label j = 0; j < sp_num; j++)
        {
            reactions_ptr_ref.v_net[i + j*react_num] = reactions_ptr_ref.vr[i + j*react_num] - reactions_ptr_ref.vf[i + j*react_num];
        }

        if (reactions_ptr_ref.react_type[i] == 0. || reactions_ptr_ref.react_type[i] == 1.)
        {
            reactions_ptr_ref.abe[6*i] = subDict_.get<scalar>("A");
            reactions_ptr_ref.abe[6*i + 1] = subDict_.get<scalar>("beta");
            reactions_ptr_ref.abe[6*i + 2] = subDict_.get<scalar>("Ta");
        }
        else
        {
            reactions_ptr_ref.abe[6*i] = subDict_.subDict("kInf").get<scalar>("A");
            reactions_ptr_ref.abe[6*i + 1] = subDict_.subDict("kInf").get<scalar>("beta");
            reactions_ptr_ref.abe[6*i + 2] = subDict_.subDict("kInf").get<scalar>("Ta");

            reactions_ptr_ref.abe[6*i + 3] = subDict_.subDict("k0").get<scalar>("A");
            reactions_ptr_ref.abe[6*i + 4] = subDict_.subDict("k0").get<scalar>("beta");
            reactions_ptr_ref.abe[6*i + 5] = subDict_.subDict("k0").get<scalar>("Ta");

        }

    }

    scalargpuField& T = this->thermo().T().primitiveFieldRef();
    scalargpuField& p = this->thermo().p().primitiveFieldRef();

    label num_cells = T.size();
    label num_species = nSpecie_;

    hipMalloc(&d_y, num_cells*num_species*sizeof(scalar));
    hipMalloc(&d_y_t, num_cells*num_species*sizeof(scalar));
    hipMalloc(&d_Ynew, num_cells*num_species*sizeof(scalar));
    hipMalloc(&d_T, num_cells*sizeof(scalar));
    hipMalloc(&d_p, num_cells*sizeof(scalar));


    forAll(Y_, speciesI) {
        volScalargpuField& Yi = Y_[speciesI];
        scalargpuField& Yi_tmp = Yi.primitiveFieldRef();
        thrust::copy(Yi_tmp.begin(),Yi_tmp.end(), thrust::device_pointer_cast(d_y + speciesI*num_cells));
    }
   

    opencc_create(&elements_const_ref, &species_const_ref, &thermo_ptr_ref, &reactions_ptr_ref);

   
    if(thermo_ptr_ref.T == nullptr)thermo_ptr_ref.T = (scalar *)malloc(num_cells*sizeof(scalar));
    thrust::copy(T.begin(),T.end(),thermo_ptr_ref.T);
    if(thermo_ptr_ref.P == nullptr)thermo_ptr_ref.P = (scalar *)malloc(num_cells*sizeof(scalar));
    thrust::copy(p.begin(),p.end(),thermo_ptr_ref.P);
    if(thermo_ptr_ref.Y == nullptr)thermo_ptr_ref.Y = (scalar *)malloc(species_const_ref.sp_num*num_cells*sizeof(scalar));
    thrust::copy(thrust::device_pointer_cast(d_y), thrust::device_pointer_cast(d_y+num_cells*num_species), thermo_ptr_ref.Y);

    set_thermoFluid(species_const_ref.sp_num, num_cells, &thermo_ptr_ref, THERMO_ALL);

    set_reactions(reactions_ptr_ref.react_num, &reactions_ptr_ref, REACTIONS_ALL);   

    Info<< "GPUChemistryModel: Number of species = " << nSpecie_
        << " and reactions = " << nReaction_ << endl;
}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

template<class ReactionThermo, class ThermoType>
Foam::GPUChemistryModel<ReactionThermo, ThermoType>::
~GPUChemistryModel()
{
    hipFree(d_y);
    hipFree(d_y_t);
    hipFree(d_Ynew);
    hipFree(d_T);
    hipFree(d_p);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class ReactionThermo, class ThermoType>
Foam::tmp<Foam::volScalargpuField>
Foam::GPUChemistryModel<ReactionThermo, ThermoType>::tc() const
{
    NotImplemented;
    return volScalargpuField::null();
}


template<class ReactionThermo, class ThermoType>
Foam::tmp<Foam::volScalargpuField>
Foam::GPUChemistryModel<ReactionThermo, ThermoType>::Qdot() const
{
    tmp<volScalargpuField> tQdot
    (
        new volScalargpuField
        (
            IOobject
            (
                "Qdot",
                this->mesh_.time().timeName(),
                this->mesh_.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            this->mesh_,
            dimensionedScalar(dimEnergy/dimVolume/dimTime, Zero)
        )
    );

    if (this->chemistry_)
    {
        scalargpuField& Qdot = tQdot.ref();
        
        forAll(Y_, i)
        {
            const scalar hi = specieThermo_[i].Hc();      
            Qdot -= hi * RR_[i].field(); 
        }
    }

    return tQdot;
}


template<class ReactionThermo, class ThermoType>
Foam::tmp<Foam::DimensionedgpuField<Foam::scalar, Foam::gpuvolMesh>>
Foam::GPUChemistryModel<ReactionThermo, ThermoType>::calculateRR
(
    const label ri,
    const label si
) const
{
    NotImplemented;
    return tmp<DimensionedgpuField<scalar, gpuvolMesh>>
    (
        DimensionedgpuField<scalar, gpuvolMesh>::null()
    );
}


template<class ReactionThermo, class ThermoType>
void Foam::GPUChemistryModel<ReactionThermo, ThermoType>::calculate()
{
    NotImplemented;
    
}


template<class ReactionThermo, class ThermoType>
template<class DeltaTType>
Foam::scalar Foam::GPUChemistryModel<ReactionThermo, ThermoType>::solve
(
    const DeltaTType& deltaT
)
{
    NotImplemented;
    return scalar(1.0);
}


template<class ReactionThermo, class ThermoType>
Foam::scalar Foam::GPUChemistryModel<ReactionThermo, ThermoType>::solve
(
    const scalar deltaT
)
{
    
    BasicChemistryModel<ReactionThermo>::correct();

    scalar deltaTMin = GREAT;

    if (!this->chemistry_)
    {
        return deltaTMin;
    }

    tmp<volScalargpuField> trho(this->thermo().rho());
    const volScalargpuField& rho_tmp = trho();
    const scalargpuField& rho = rho_tmp.primitiveField();

    scalargpuField& T = this->thermo().T().primitiveFieldRef();
    scalargpuField& p = this->thermo().p().primitiveFieldRef();

    label num_cells = T.size();

    thrust::copy(T.begin(),T.end(), thrust::device_pointer_cast(d_T));
    thrust::copy(p.begin(),p.end(), thrust::device_pointer_cast(d_p));

    scalar t_init = this->deltaTChemIni_;
    scalar t_end = deltaT;
    
    this->solve(d_Ynew, Y_, d_T, d_p, t_init, t_end, num_cells);

    for (label i = 0; i < nSpecie_; i++)
    {
        volScalargpuField::Internal& RR_tmp = RR_[i];
        scalargpuField& RR_tmp2 = RR_tmp.field();
        scalargpuField Ynew_tmp(num_cells);
        thrust::copy_n(thrust::device_pointer_cast(d_Ynew+i*num_cells),num_cells,Ynew_tmp.begin());
        RR_tmp2 = (Ynew_tmp - Y_[i].primitiveFieldRef())*rho/deltaT;
    }
    

    return 0.0;

}


template<class ReactionThermo, class ThermoType>
Foam::scalar Foam::GPUChemistryModel<ReactionThermo, ThermoType>::solve
(
    const scalargpuField& deltaT
)
{
    //return this->solve<scalarField>(deltaT);
    NotImplemented
    return scalar(1.0);

}


// ************************************************************************* //
