/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::pureMixture

Description
    Foam::pureMixture

SourceFiles
    pureMixture.C

\*---------------------------------------------------------------------------*/

#ifndef pureMixture_H
#define pureMixture_H

#include "basicMixture.H"
#include <thrust/device_ptr.h>
#include <thrust/device_malloc.h>
#include <thrust/device_free.h>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                         Class pureMixture Declaration
\*---------------------------------------------------------------------------*/

template<class ThermoType>
class pureMixture
:
    public basicMixture
{
    // Private data

        const gpufvMesh& mesh_;
		ThermoType mixture_;
        mutable thrust::device_ptr<ThermoType> mixtureCells_;
		mutable thrust::device_ptr<ThermoType> mixturePatchs_;

        //- Construct as copy (not implemented)
        pureMixture(const pureMixture<ThermoType>&) = delete;


public:

    //- The type of thermodynamics this mixture is instantiated for
    typedef ThermoType thermoType;


    // Constructors

        //- Construct from dictionary, mesh and phase name
        pureMixture(const dictionary&, const gpufvMesh&, const word&);


    // Member functions

        //- Return the instantiated type name
        static word typeName()
        {
            return "pureMixture<" + ThermoType::typeName() + '>';
        }

        const ThermoType& mixture() const
        {
            return mixture_;
        }

		__host__ __device__
        const ThermoType& cellMixture(const label) const
        {
            return mixture_;
        }

		__host__ __device__
        const ThermoType& patchFaceMixture
        (
            const label,
            const label
        ) const
        {
            return mixture_;
        }

        const ThermoType& cellVolMixture
        (
            const scalar,
            const scalar,
            const label
        ) const
        {
            return mixture_;
        }

        const ThermoType& patchFaceVolMixture
        (
            const scalar,
            const scalar,
            const label,
            const label
        ) const
        {
            return mixture_;
        }

        //- Read dictionary
        void read(const dictionary&);

        void updateCellMixture()
        {
        }

        const thrust::device_ptr<ThermoType>& updatePatchFaceMixture(const label patchi) const
        {
			if (mixturePatchs_)
			{
				thrust::device_free(mixturePatchs_);
			}

			mixturePatchs_ = thrust::device_malloc<ThermoType>(mesh_.boundary()[patchi].size());
			thrust::fill(mixturePatchs_,mixturePatchs_+mesh_.boundary()[patchi].size(),mixture_);
			
            return mixturePatchs_;
        }

        void updateCellVolMixture
        (
            const scalargpuField& pCells,
            const scalargpuField& TCells
        )
        {
        }

        const thrust::device_ptr<ThermoType>& updatePatchFaceVolMixture
        (
            const scalargpuField& pCells,
            const scalargpuField& TCells,
            const label patchi
        ) const
        {
			if (mixturePatchs_)
			{
				thrust::device_free(mixturePatchs_);
			}

			mixturePatchs_ = thrust::device_malloc<ThermoType>(mesh_.boundary()[patchi].size());
			thrust::fill(mixturePatchs_,mixturePatchs_+mesh_.boundary()[patchi].size(),mixture_);
            return mixturePatchs_;
        }
		
        const thrust::device_ptr<ThermoType>& mixtureCells() const
       	{
        	return mixtureCells_;
		}
		
        const thrust::device_ptr<ThermoType>& mixtureVolCells() const
       	{
        	return mixtureCells_;
		}
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#ifdef NoRepository
    #include "pureMixture.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
