/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpusurfaceInterpolate.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type>
Foam::tmp<Foam::gpusurfaceInterpolationScheme<Type>>
Foam::fvc::scheme
(
    const surfaceScalargpuField& faceFlux,
    Istream& streamData
)
{
    return gpusurfaceInterpolationScheme<Type>::New
    (
        faceFlux.mesh(),
        faceFlux,
        streamData
    );
}


template<class Type>
Foam::tmp<Foam::gpusurfaceInterpolationScheme<Type>> Foam::fvc::scheme
(
    const surfaceScalargpuField& faceFlux,
    const word& name
)
{
    return gpusurfaceInterpolationScheme<Type>::New
    (
        faceFlux.mesh(),
        faceFlux,
        faceFlux.mesh().hostmesh().interpolationScheme(name)
    );
}


template<class Type>
Foam::tmp<Foam::gpusurfaceInterpolationScheme<Type>> Foam::fvc::scheme
(
    const gpufvMesh& mesh,
    Istream& streamData
)
{
    return gpusurfaceInterpolationScheme<Type>::New
    (
        mesh,
        streamData
    );
}


template<class Type>
Foam::tmp<Foam::gpusurfaceInterpolationScheme<Type>> Foam::fvc::scheme
(
    const gpufvMesh& mesh,
    const word& name
)
{
    return gpusurfaceInterpolationScheme<Type>::New
    (
        mesh,
        mesh.hostmesh().interpolationScheme(name)
    );
}


template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    const surfaceScalargpuField& faceFlux,
    Istream& schemeData
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << endl;
    }

    return scheme<Type>(faceFlux, schemeData)().interpolate(vf);
}


template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    const surfaceScalargpuField& faceFlux,
    const word& name
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << " using " << name << endl;
    }

    return scheme<Type>(faceFlux, name)().interpolate(vf);
}

template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf,
    const surfaceScalargpuField& faceFlux,
    const word& name
)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsf =
        interpolate(tvf(), faceFlux, name);

    tvf.clear();

    return tsf;
}

template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    const tmp<surfaceScalargpuField>& tFaceFlux,
    const word& name
)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsf =
        interpolate(vf, tFaceFlux(), name);

    tFaceFlux.clear();

    return tsf;
}

template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf,
    const tmp<surfaceScalargpuField>& tFaceFlux,
    const word& name
)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsf =
        interpolate(tvf(), tFaceFlux(), name);

    tvf.clear();
    tFaceFlux.clear();

    return tsf;
}


template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    Istream& schemeData
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << endl;
    }

    return scheme<Type>(vf.mesh(), schemeData)().interpolate(vf);
}

template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    const word& name
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << " using " << name
            << endl;
    }

    return scheme<Type>(vf.mesh(), name)().interpolate(vf);
}

template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf,
    const word& name
)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsf =
        interpolate(tvf(), name);

    tvf.clear();

    return tsf;
}


template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << " using run-time selected scheme"
            << endl;
    }

    return interpolate(vf, "interpolate(" + vf.name() + ')');
}


template<class Type>
Foam::tmp<Foam::GeometricgpuField<Type, Foam::fvsPatchgpuField, Foam::gpusurfaceMesh>>
Foam::fvc::interpolate
(
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf
)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsf =
        interpolate(tvf());
    tvf.clear();
    return tsf;
}


template<class Type>
Foam::tmp<Foam::FieldField<Foam::fvsPatchgpuField, Type>>
Foam::fvc::interpolate
(
    const FieldField<fvPatchgpuField, Type>& fvpff
)
{
    FieldField<fvsPatchgpuField, Type>* fvspffPtr
    (
        new FieldField<fvsPatchgpuField, Type>(fvpff.size())
    );

    forAll(*fvspffPtr, patchi)
    {
        fvspffPtr->set
        (
            patchi,
            fvsPatchgpuField<Type>::NewCalculatedType(fvpff[patchi].patch()).ptr()
        );
        (*fvspffPtr)[patchi] = fvpff[patchi];
    }

    return tmp<FieldField<fvsPatchgpuField, Type>>(fvspffPtr);
}


template<class Type>
Foam::tmp<Foam::FieldField<Foam::fvsPatchgpuField, Type>>
Foam::fvc::interpolate
(
    const tmp<FieldField<fvPatchgpuField, Type>>& tfvpff
)
{
    tmp<FieldField<fvsPatchgpuField, Type>> tfvspff = interpolate(tfvpff());
    tfvpff.clear();
    return tfvspff;
}


template<class Type>
Foam::tmp
<
    Foam::GeometricgpuField
    <
        typename Foam::innerProduct<Foam::vector, Type>::type,
        Foam::fvsPatchgpuField,
        Foam::gpusurfaceMesh
    >
>
Foam::fvc::dotInterpolate
(
    const surfaceVectorgpuField& Sf,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    if (gpusurfaceInterpolation::debug)
    {
        InfoInFunction
            << "interpolating GeometricField<Type, fvPatchField, volMesh> "
            << vf.name() << " using run-time selected scheme"
            << endl;
    }

    return scheme<Type>
    (
        vf.mesh(),
        "dotInterpolate(" + Sf.name() + ',' + vf.name() + ')'
    )().dotInterpolate(Sf, vf);
}


template<class Type>
Foam::tmp
<
    Foam::GeometricgpuField
    <
        typename Foam::innerProduct<Foam::vector, Type>::type,
        Foam::fvsPatchgpuField,
        Foam::gpusurfaceMesh
    >
>
Foam::fvc::dotInterpolate
(
    const surfaceVectorgpuField& Sf,
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf
)
{
    tmp
    <
        GeometricgpuField
        <
            typename Foam::innerProduct<Foam::vector, Type>::type,
            fvsPatchgpuField,
            gpusurfaceMesh
        >
    > tsf = dotInterpolate(Sf, tvf());
    tvf.clear();
    return tsf;
}


// ************************************************************************* //
