/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::localMin

Group
    grpFvSurfaceInterpolationSchemes

Description
    LocalMin-mean differencing scheme class.

    This scheme interpolates 1/field using a scheme specified at run-time
    and return the reciprocal of the interpolate.

SourceFiles
    localMin.C

\*---------------------------------------------------------------------------*/

#ifndef gpulocalMin_H
#define gpulocalMin_H

#include "gpusurfaceInterpolationScheme.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class localMin Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulocalMin
:
    public gpusurfaceInterpolationScheme<Type>
{
    // Private Member Functions

        //- No copy assignment
        void operator=(const gpulocalMin&) = delete;


public:

    //- Runtime type information
    TypeName("localMin");


    // Constructors

        //- Construct from mesh
        gpulocalMin(const gpufvMesh& mesh)
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}

        //- Construct from Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpulocalMin
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}

        //- Construct from faceFlux and Istream
        gpulocalMin
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}


    // Member Functions

        //- Return the interpolation weighting factors
        virtual tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const
        {
            NotImplemented;

            return tmp<surfaceScalargpuField>(nullptr);
        }

        //- Return the face-interpolate of the given cell field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            const gpufvMesh& mesh = vf.mesh();

            tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tvff
            (
                new GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>
                (
                    IOobject
                    (
                        "gpulocalMin::interpolate(" + vf.name() + ')',
                        mesh.time().timeName(),
                        mesh.hostmesh()
                    ),
                    mesh,
                    vf.dimensions()
                )
            );
            GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& vff = tvff.ref();

            const labelgpuList& own = mesh.owner();
            const labelgpuList& nei = mesh.neighbour();

            /*forAll(vff, facei)
            {
                vff[facei] = min(vf[own[facei]], vf[nei[facei]]);
            }*/

            thrust::transform(thrust::make_permutation_iterator(vf.field().begin(),own.begin()),
                              thrust::make_permutation_iterator(vf.field().begin(),own.end()),
                              thrust::make_permutation_iterator(vf.field().begin(),nei.begin()),
                              vff.field().begin(),
                              minModBinaryFunctionFunctor<Type,Type,Type>());

            typename GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>::
                Boundary& bff = vff.boundaryFieldRef();

            forAll(bff, patchi)
            {
                const fvPatchgpuField<Type>& pf = vf.boundaryField()[patchi];
                gpuField<Type>& pff = bff[patchi];

                if (pf.coupled())
                {
                    tmp<gpuField<Type>> tpif(pf.patchInternalField());
                    const gpuField<Type>& pif = tpif();

                    tmp<gpuField<Type>> tpnf(pf.patchNeighbourField());
                    const gpuField<Type>& pnf = tpnf();

                    /*forAll(pff, i)
                    {
                        pff[i] = min(pif[i], pnf[i]);
                    }*/
                    thrust::transform(
                        pif.begin(),pif.end(),
                        pnf.begin(),
                        pff.begin(),
                        minModBinaryFunctionFunctor<Type,Type,Type>()
                    );
                }
                else
                {
                    pff = pf;
                }
            }

            return tvff;
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
