/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::linearUpwind

Group
    grpFvSurfaceInterpolationSchemes

Description
    linearUpwind interpolation scheme class derived from upwind and returns
    upwind weighting factors and also applies a gradient-based explicit
    correction.

SourceFiles
    linearUpwind.C

\*---------------------------------------------------------------------------*/

#ifndef gpulinearUpwind_H
#define gpulinearUpwind_H

#include "gpuupwind.H"
#include "gpugaussGrad.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class linearUpwind Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulinearUpwind
:
    public gpuupwind<Type>
{
    // Private Data

        word gradSchemeName_;


    // Private Member Functions

        //- No copy construct
        gpulinearUpwind(const gpulinearUpwind&) = delete;

        //- No copy assignment
        void operator=(const gpulinearUpwind&) = delete;


public:

    //- Runtime type information
    TypeName("linearUpwind");


    // Constructors

        //- Construct from faceFlux
        gpulinearUpwind
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux
        )
        :
            gpuupwind<Type>(mesh, faceFlux),
            gradSchemeName_("grad")
        {}

        //- Construct from Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpulinearUpwind
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        )
        :
            gpuupwind<Type>(mesh, schemeData),
            gradSchemeName_(schemeData)
        {}

        //- Construct from faceFlux and Istream
        gpulinearUpwind
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        )
        :
            gpuupwind<Type>(mesh, faceFlux, schemeData),
            gradSchemeName_(schemeData)
        {}


    // Member Functions

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const
        {
            return true;
        }

        //- Return the explicit correction to the face-interpolate
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;
};


//// Specialize for volVectorField to support cached gradients
//template<>
//tmp<surfaceVectorgpuField> gpulinearUpwind<vector>::correction
//(
//    const volVectorgpuField& vf
//) const;


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
