/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::multivariateScheme

Description
    Generic multi-variate discretisation scheme class which may be instantiated
    for any of the NVD, CNVD or NVDV schemes.

SourceFiles
    multivariateScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpumultivariateScheme_H
#define gpumultivariateScheme_H

#include "gpumultivariateSurfaceInterpolationScheme.H"
#include "surfacegpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class multivariateScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type, class Scheme>
class gpumultivariateScheme
:
    public gpumultivariateSurfaceInterpolationScheme<Type>,
    public Scheme::LimiterType
{
    // Private data

        const surfaceScalargpuField& faceFlux_;
        surfaceScalargpuField weights_;


    // Private Member Functions

        //- No copy construct
        gpumultivariateScheme(const gpumultivariateScheme&) = delete;

        //- No copy assignment
        void operator=(const gpumultivariateScheme&) = delete;


public:

    //- Runtime type information
    TypeName("gpumultivariateScheme");


    // Constructors

        //- Construct for field, faceFlux and Istream
        gpumultivariateScheme
        (
            const gpufvMesh& mesh,
            const typename gpumultivariateSurfaceInterpolationScheme<Type>::
                fieldTable& fields,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    // Member Operators

        //- surfaceInterpolationScheme sub-class returned by operator(field)
        //  which is used as the interpolation scheme for the field
        class fieldScheme
        :
            public gpumultivariateSurfaceInterpolationScheme<Type>::
                fieldScheme
        {
            // Private data

                const surfaceScalargpuField& weights_;

        public:

            // Constructors

                fieldScheme
                (
                    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
                    const surfaceScalargpuField& weights
                )
                :
                    gpumultivariateSurfaceInterpolationScheme<Type>::
                        fieldScheme(field),
                    weights_(weights)
                {}


            // Member Functions

                //- Return the interpolation weighting factors
                tmp<surfaceScalargpuField> weights
                (
                    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
                ) const
                {
                    return weights_;
                }
        };

        tmp<gpusurfaceInterpolationScheme<Type>> operator()
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
        ) const
        {
            return tmp<gpusurfaceInterpolationScheme<Type>>
            (
                new fieldScheme(field, weights_)
            );
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeLimitedMultivariateSurfaceInterpolationScheme(SS, LIMITER)         \
typedef gpumultivariateScheme                                                     \
<                                                                              \
    scalar,                                                                    \
    gpuLimitedScheme<scalar, LIMITER<gpuNVDTVD>, gpulimitFuncs::magSqr>                 \
>                                                                              \
    gpumultivariateScheme##LIMITER##_;                                            \
    defineTemplateTypeNameAndDebugWithName                                     \
    (                                                                          \
        gpumultivariateScheme##LIMITER##_,                                        \
        #SS,                                                                   \
        0                                                                      \
    );                                                                         \
                                                                               \
gpumultivariateSurfaceInterpolationScheme<scalar>::addIstreamConstructorToTable   \
<                                                                              \
    gpumultivariateScheme                                                         \
    <                                                                          \
        scalar,                                                                \
        gpuLimitedScheme<scalar, LIMITER<gpuNVDTVD>, gpulimitFuncs::magSqr>             \
    >                                                                          \
>                                                                              \
    addgpuMultivariate##SS##ConstructorToTable_;


#define makeLLimitedMultivariateSurfaceInterpolationScheme\
(                                                                              \
    SS,                                                                        \
    LLIMITER,                                                                  \
    LIMITER,                                                                   \
    NVDTVD,                                                                    \
    LIMFUNC                                                                    \
)                                                                              \
typedef gpumultivariateScheme                                                     \
<                                                                              \
    scalar,                                                                    \
    gpuLimitedScheme<scalar, LLIMITER<LIMITER<NVDTVD>>, gpulimitFuncs::LIMFUNC>      \
>                                                                              \
    gpumultivariateScheme##LLIMITER##LIMITER##NVDTVD##LIMFUNC##_;                 \
    defineTemplateTypeNameAndDebugWithName                                     \
    (                                                                          \
        gpumultivariateScheme##LLIMITER##LIMITER##NVDTVD##LIMFUNC##_,             \
        #SS,                                                                   \
        0                                                                      \
    );                                                                         \
                                                                               \
gpumultivariateSurfaceInterpolationScheme<scalar>::addIstreamConstructorToTable   \
<                                                                              \
    gpumultivariateScheme                                                         \
    <                                                                          \
        scalar,                                                                \
        gpuLimitedScheme<scalar, LLIMITER<LIMITER<NVDTVD>>, gpulimitFuncs::LIMFUNC>  \
    >                                                                          \
>                                                                              \
    addgpuMultivariate##SS##ConstructorToTable_;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpumultivariateScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
