/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::AssemblyFvPatch

Description
    An assembly of lduMatrix that is specific inter-region coupling
    through mapped patches.

SourceFiles
    AssemblyFvPatch.C

\*---------------------------------------------------------------------------*/

#ifndef AssemblygpuFvPatch_H
#define AssemblygpuFvPatch_H


#include "cyclicgpuFvPatch.H"
#include "cyclicAMIgpuFvPatch.H"

#include <thrust/iterator/permutation_iterator.h>
#include <thrust/copy.h>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                    Class AssemblyFvPatch Declaration
\*---------------------------------------------------------------------------*/

template<class CyclicType>
class AssemblygpuFvPatch
:
    public CyclicType
{
protected:

    // Protected Data

        //- Nbr faceCell
        const labelgpuList& nrbFaceCells_;

        //- Local faceCells
        const labelgpuList faceCells_;

        //- Nbr patch ID
        label nbrPatchID_;

        //- Non-overlap patch ID
        label nonOverlapPatchID_;


public:

    //- Runtime type information
    TypeName(CyclicType::typeName_());


    // Constructors

        //- Construct from components
        AssemblygpuFvPatch
        (
            const polyPatch& patch,
            const gpufvBoundaryMesh& bm,
            const labelgpuList& nrbFaceCells,
            const labelgpuList& faceCells,
            const label nbrPatchID,
            const label nonOverPatchID = -1
        )
        :
            CyclicType(patch, bm),
            nrbFaceCells_(nrbFaceCells),
            faceCells_(faceCells),
            nbrPatchID_(nbrPatchID),
            nonOverlapPatchID_(nonOverPatchID)
        {}


    //- Destructor
    virtual ~AssemblygpuFvPatch() = default;


    // Member Functions

        //- Return neighbour field
        virtual tmp<labelgpuField> internalFieldTransfer
        (
            const Pstream::commsTypes commsType,
            const labelgpuList& internalData
        ) const
        {
            auto tpif = tmp<labelgpuField>::New(nrbFaceCells_.size());
            auto& pif = tpif.ref();

			thrust::copy
			(
			thrust::make_permutation_iterator
	        (
            internalData.begin(),
            nrbFaceCells_.begin()
			),
	        thrust::make_permutation_iterator
	        (
            internalData.begin(),
            nrbFaceCells_.end()
	        ),
	        pif.begin()
	        );

            return tpif;
        }

        //- Return neighbour
        virtual label neighbPatchID() const
        {
            return nbrPatchID_;
        }

        //- Return non-overlap patch ID for ACMI
        virtual label nonOverlapPatchID() const
        {
            return nonOverlapPatchID_;
        }

        //- Return non-overlapping fvPatch for ACMI
        virtual const gpufvPatch& nonOverlapPatch() const
        {
            return this->boundaryMesh()[nonOverlapPatchID_];
        }

        //- Return nbr patch
        virtual const CyclicType& neighbPatch() const
        {
            return refCast<const CyclicType>
            (
                this->boundaryMesh()[nbrPatchID_]
            );
        }

        //- Return faceCells

        virtual const labelgpuList& gpuFaceCells() const
        {
            return faceCells_;
        }
};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#define makeAssemblygpuFvPatch(Type)                                             \
                                                                              \
    defineNamedTemplateTypeNameAndDebug(AssemblygpuFvPatch<Type>, 0);            \


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
