/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2015 OpenFOAM Foundation
    Copyright (C) 2016-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpuMULES.H"
#include "profiling.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
	struct limitSumlambdaPosFunctor{

		__host__ __device__
		scalar operator()(const scalar& sumPos, const scalar& sumNeg)
		{
			if(sumPos+sumNeg>0 && sumPos>VSMALL) return -sumNeg/sumPos;
		}
	};

	struct limitSumlambdaNegFunctor{

		__host__ __device__
		scalar operator()(const scalar& sumPos, const scalar& sumNeg)
		{
			if(sumPos+sumNeg<0 && sumNeg<-VSMALL) return -sumPos/sumNeg;
		}
	};

	struct limitSumPhiPsiPosFunctor{

		__host__ __device__
		scalar operator()(const scalar& lambda, const scalar& phiPsi)
		{
			if(phiPsi>0) return lambda*phiPsi;
		}
	};

	struct limitSumPhiPsiNegFunctor{

		__host__ __device__
		scalar operator()(const scalar& lambda, const scalar& phiPsi)
		{
			if(phiPsi<0) return lambda*phiPsi;
		}
	};
}

void Foam::MULES::gpulimitSum(UPtrList<scalargpuField>& phiPsiCorrs)
{
    const label nPhases = phiPsiCorrs.size();

//    forAll(phiPsiCorrs[0], facei)
//    {
//        scalar sumPos = 0;
//        scalar sumNeg = 0;
//
//        for (label phasei = 0; phasei < nPhases; ++phasei)
//        {
//            if (phiPsiCorrs[phasei][facei] > 0)
//            {
//                sumPos += phiPsiCorrs[phasei][facei];
//            }
//            else
//            {
//                sumNeg += phiPsiCorrs[phasei][facei];
//            }
//        }
//
//        scalar sum = sumPos + sumNeg;
//
//        if (sum > 0 && sumPos > VSMALL)
//        {
//            scalar lambda = -sumNeg/sumPos;
//
//            for (label phasei = 0; phasei < nPhases; ++phasei)
//            {
//                if (phiPsiCorrs[phasei][facei] > 0)
//                {
//                    phiPsiCorrs[phasei][facei] *= lambda;
//                }
//            }
//        }
//        else if (sum < 0 && sumNeg < -VSMALL)
//        {
//            scalar lambda = -sumPos/sumNeg;
//
//            for (label phasei = 0; phasei < nPhases; ++phasei)
//            {
//                if (phiPsiCorrs[phasei][facei] < 0)
//                {
//                    phiPsiCorrs[phasei][facei] *= lambda;
//                }
//            }
//        }
//    }

	scalargpuField sumPos(phiPsiCorrs[0].size(), 0.0);
	scalargpuField sumNeg(phiPsiCorrs[0].size(), 0.0);
	for (label phasei = 0; phasei < nPhases; ++phasei)
	{
		sumPos += max(phiPsiCorrs[phasei], 0.0);
		sumNeg += min(phiPsiCorrs[phasei], 0.0);
	}

	scalargpuField lambda(phiPsiCorrs[0].size(),1.0);

	thrust::transform(sumPos.begin(),sumPos.end(),sumNeg.begin(),lambda.begin(),limitSumlambdaPosFunctor());
	
	for (label phasei = 0; phasei < nPhases; ++phasei)
	{
		thrust::transform
		(
			lambda.begin(),
			lambda.end(),
			phiPsiCorrs[phasei].begin(),
			phiPsiCorrs[phasei].begin(),
			limitSumPhiPsiPosFunctor()
		);
	}

	lambda = 1.0;
	
	thrust::transform(sumPos.begin(),sumPos.end(),sumNeg.begin(),lambda.begin(),limitSumlambdaNegFunctor());
	
	for (label phasei = 0; phasei < nPhases; ++phasei)
	{
		thrust::transform
		(
			lambda.begin(),
			lambda.end(),
			phiPsiCorrs[phasei].begin(),
			phiPsiCorrs[phasei].begin(),
			limitSumPhiPsiNegFunctor()
		);
	}
}


void Foam::MULES::gpulimitSum
(
    const UPtrList<const scalargpuField>& alphas,
    UPtrList<scalargpuField>& phiPsiCorrs,
    const labelHashSet& fixed
)
{
    labelHashSet notFixed(identity(phiPsiCorrs.size()));
    notFixed -= fixed;

//    forAll(phiPsiCorrs[0], facei)
//    {
//        scalar alphaNotFixed = 0, corrNotFixed = 0;
//        for (const label phasei : notFixed)
//        {
//            alphaNotFixed += alphas[phasei][facei];
//            corrNotFixed += phiPsiCorrs[phasei][facei];
//        }
//
//        scalar corrFixed = 0;
//        for (const label phasei : fixed)
//        {
//            corrFixed += phiPsiCorrs[phasei][facei];
//        }
//
//        const scalar sumCorr = corrNotFixed + corrFixed;
//
//        const scalar lambda = - sumCorr/alphaNotFixed;
//
//        for (const label phasei : notFixed)
//        {
//            phiPsiCorrs[phasei][facei] += lambda*alphas[phasei][facei];
//        }
//    }

	scalargpuField alphaNotFixed(phiPsiCorrs[0].size(), 0.0);
	scalargpuField corrNotFixed(phiPsiCorrs[0].size(), 0.0);

	for (const label phasei : notFixed)
	{
		alphaNotFixed += alphas[phasei];
		corrNotFixed += phiPsiCorrs[phasei];
	}

	scalargpuField corrFixed(phiPsiCorrs[0].size(), 0.0);
	for (const label phasei : fixed)
	{
		corrFixed += phiPsiCorrs[phasei];
	}
	
	const scalargpuField sumCorr = corrNotFixed + corrFixed;
	const scalargpuField lambda = - sumCorr/alphaNotFixed;

	for (const label phasei : notFixed)
	{
		phiPsiCorrs[phasei] += lambda*alphas[phasei];
	}
	
}

// ************************************************************************* //
