/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "variableHeightFlowRateFvPatchgpuField.H"
#include "fvPatchgpuFieldMapper.H"
#include "addToRunTimeSelectionTable.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::variableHeightFlowRateFvPatchScalargpuField
::variableHeightFlowRateFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    mixedFvPatchgpuField<scalar>(p, iF),
    phiName_("phi"),
    lowerBound_(0.0),
    upperBound_(1.0)
{
    this->refValue() = 0.0;
    this->refGrad() = 0.0;
    this->valueFraction() = 0.0;
}


Foam::variableHeightFlowRateFvPatchScalargpuField
::variableHeightFlowRateFvPatchScalargpuField
(
    const variableHeightFlowRateFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    mixedFvPatchScalargpuField(ptf, p, iF, mapper),
    phiName_(ptf.phiName_),
    lowerBound_(ptf.lowerBound_),
    upperBound_(ptf.upperBound_)
{}


Foam::variableHeightFlowRateFvPatchScalargpuField
::variableHeightFlowRateFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    mixedFvPatchScalargpuField(p, iF),
    phiName_(dict.getOrDefault<word>("phi", "phi")),
    lowerBound_(dict.get<scalar>("lowerBound")),
    upperBound_(dict.get<scalar>("upperBound"))
{
    patchType() = dict.getOrDefault<word>("patchType", word::null);
    this->refValue() = 0.0;

    if (dict.found("value"))
    {
        fvPatchScalargpuField::operator=
        (
            scalargpuField("value", dict, p.size())
        );
    }
    else
    {
        fvPatchScalargpuField::operator=(this->patchInternalField());
    }

    this->refGrad() = 0.0;
    this->valueFraction() = 0.0;
}


Foam::variableHeightFlowRateFvPatchScalargpuField
    ::variableHeightFlowRateFvPatchScalargpuField
(
    const variableHeightFlowRateFvPatchScalargpuField& ptf
)
:
    mixedFvPatchScalargpuField(ptf),
    phiName_(ptf.phiName_),
    lowerBound_(ptf.lowerBound_),
    upperBound_(ptf.upperBound_)
{}


Foam::variableHeightFlowRateFvPatchScalargpuField
    ::variableHeightFlowRateFvPatchScalargpuField
(
    const variableHeightFlowRateFvPatchScalargpuField& ptf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    mixedFvPatchScalargpuField(ptf, iF),
    phiName_(ptf.phiName_),
    lowerBound_(ptf.lowerBound_),
    upperBound_(ptf.upperBound_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //
namespace Foam
{
    struct variableHeightFlowRateUpdateCoeffs{
        const scalar lowerBound;
        const scalar upperBound;
        variableHeightFlowRateUpdateCoeffs(scalar l, scalar u): lowerBound(1), upperBound(u){}
        __host__ __device__
        thrust::tuple<scalar,scalar> operator()(const scalar& phip, const scalar& alphap){
            scalar val;
            scalar frac;

            if (phip < -SMALL)
            {
                if (phip < -SMALL)
                {
                    val = 0.0;
                }
                else if (alphap > upperBound)
                {
                    val = 1.0;
                }
                else
                {
                    val = alphap;
                }

                frac = 1.0;
            }
            else
            {
                val = 0.0;
                frac = 0.0;
            }
            return thrust::make_tuple(val,frac);
        }
    };
}

void Foam::variableHeightFlowRateFvPatchScalargpuField::updateCoeffs()
{
    if (this->updated())
    {
        return;
    }

    const fvsPatchgpuField<scalar>& phip =
        patch().lookupPatchField<surfaceScalargpuField, scalar>(phiName_);

    scalargpuField alphap(this->patchInternalField());

    thrust::transform(phip.begin(),phip.end(),alphap.begin(),
                      thrust::make_zip_iterator(thrust::make_tuple(this->refValue().begin(),this->valueFraction().begin())),
                      variableHeightFlowRateUpdateCoeffs(lowerBound_,upperBound_));

    mixedFvPatchScalargpuField::updateCoeffs();
}


void Foam::variableHeightFlowRateFvPatchScalargpuField::write(Ostream& os) const
{
    fvPatchScalargpuField::write(os);
    os.writeEntryIfDifferent<word>("phi", "phi", phiName_);
    os.writeEntry("lowerBound", lowerBound_);
    os.writeEntry("upperBound", upperBound_);
    this->writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        variableHeightFlowRateFvPatchScalargpuField
    );
}

// ************************************************************************* //
