/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::turbulentIntensityKineticEnergyInletFvPatchScalarField

Group
    grpInletBoundaryConditions

Description
    This boundary condition provides a turbulent kinetic energy condition,
    based on user-supplied turbulence intensity, defined as a fraction of the
    mean velocity:

        \f[
            k_p = 1.5 (I |U|)^2
        \f]

    where

    \vartable
        k_p     | kinetic energy at the patch
        I       | turbulence intensity
        U       | velocity field
    \endvartable

    In the event of reverse flow, a zero-gradient condition is applied.

Usage
    \table
        Property     | Description             | Required    | Default value
        intensity    | fraction of mean field [0-1] | yes    |
        U            | velocity field name     | no          | U
        phi          | flux field name         | no          | phi
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type        turbulentIntensityKineticEnergyInlet;
        intensity   0.05;           // 5% turbulence
        value       uniform 1;      // placeholder
    }
    \endverbatim

See also
    Foam::inletOutletFvPatchField

SourceFiles
    turbulentIntensityKineticEnergyInletFvPatchScalarField.C

\*---------------------------------------------------------------------------*/

#ifndef turbulentIntensityKineticEnergyInletFvPatchScalargpuField_H
#define turbulentIntensityKineticEnergyInletFvPatchScalargpuField_H

#include "inletOutletFvPatchgpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
   Class turbulentIntensityKineticEnergyInletFvPatchScalarField Declaration
\*---------------------------------------------------------------------------*/

class turbulentIntensityKineticEnergyInletFvPatchScalargpuField
:
    public inletOutletFvPatchScalargpuField
{
    // Private data

        //- Turbulent intensity as fraction of mean velocity
        scalar intensity_;

        //- Name of the velocity field
        word UName_;


public:

    //- Runtime type information
    TypeName("turbulentIntensityKineticEnergyInlet");


    // Constructors

        //- Construct from patch and internal field
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given
        //  turbulentIntensityKineticEnergyInletFvPatchScalarField
        //  onto a new patch
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
        (
            const turbulentIntensityKineticEnergyInletFvPatchScalargpuField&,
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
        (
            const turbulentIntensityKineticEnergyInletFvPatchScalargpuField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchScalargpuField> clone() const
        {
            return tmp<fvPatchScalargpuField>
            (
                new turbulentIntensityKineticEnergyInletFvPatchScalargpuField
                (
                    *this
                )
            );
        }

        //- Construct as copy setting internal field reference
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
        (
            const turbulentIntensityKineticEnergyInletFvPatchScalargpuField&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchScalargpuField> clone
        (
            const DimensionedgpuField<scalar, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchScalargpuField>
            (
                new turbulentIntensityKineticEnergyInletFvPatchScalargpuField
                (
                    *this,
                    iF
                )
            );
        }


    // Member functions

        //- Update the coefficients associated with the patch field
        virtual void updateCoeffs();

        //- Write
        virtual void write(Ostream&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
