/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::pressureInletOutletVelocityFvPatchVectorField

Group
    grpInletBoundaryConditions grpOutletBoundaryConditions

Description
    This velocity inlet/outlet boundary condition is applied to velocity
    boundaries where the pressure is specified.  A zero-gradient condition is
    applied for outflow (as defined by the flux); for inflow, the velocity is
    obtained from the patch-face normal component of the internal-cell value.

    The tangential patch velocity can be optionally specified.

Usage
    \table
        Property     | Description             | Required    | Default value
        phi          | flux field name         | no          | phi
        tangentialVelocity | tangential velocity field | no  |
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            pressureInletOutletVelocity;
        phi             phi;
        tangentialVelocity uniform (0 0 0);
        value           uniform (0 0 0);
    }
    \endverbatim

Note
    Sign conventions:
    - positive flux (out of domain): apply zero-gradient condition
    - negative flux (into of domain): derive from the flux in the patch-normal
      direction

SourceFiles
    pressureInletOutletVelocityFvPatchVectorField.C

\*---------------------------------------------------------------------------*/

#ifndef pressureInletOutletVelocityFvPatchVectorgpuField_H
#define pressureInletOutletVelocityFvPatchVectorgpuField_H

#include "fvPatchgpuFields.H"
#include "directionMixedFvPatchgpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
       Class pressureInletOutletVelocityFvPatchVectorField Declaration
\*---------------------------------------------------------------------------*/

class pressureInletOutletVelocityFvPatchVectorgpuField
:
    public directionMixedFvPatchVectorgpuField
{
    // Private data

        //- Flux field name
        word phiName_;

        //- Optional tangential velocity component
        vectorgpuField tangentialVelocity_;


public:

    //- Runtime type information
    TypeName("pressureInletOutletVelocity");


    // Constructors

        //- Construct from patch and internal field
        pressureInletOutletVelocityFvPatchVectorgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        pressureInletOutletVelocityFvPatchVectorgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given
        //  pressureInletOutletVelocityFvPatchVectorField onto a new patch
        pressureInletOutletVelocityFvPatchVectorgpuField
        (
            const pressureInletOutletVelocityFvPatchVectorgpuField&,
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        pressureInletOutletVelocityFvPatchVectorgpuField
        (
            const pressureInletOutletVelocityFvPatchVectorgpuField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchVectorgpuField> clone() const
        {
            return tmp<fvPatchVectorgpuField>
            (
                new pressureInletOutletVelocityFvPatchVectorgpuField(*this)
            );
        }

        //- Construct as copy setting internal field reference
        pressureInletOutletVelocityFvPatchVectorgpuField
        (
            const pressureInletOutletVelocityFvPatchVectorgpuField&,
            const DimensionedgpuField<vector, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchVectorgpuField> clone
        (
            const DimensionedgpuField<vector, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchVectorgpuField>
            (
                new pressureInletOutletVelocityFvPatchVectorgpuField(*this, iF)
            );
        }


    // Member functions

        // Attributes

            //- Return true: this patch field is altered by assignment
            virtual bool assignable() const
            {
                return true;
            }


        // Access

            //- Return the name of phi
            const word& phiName() const
            {
                return phiName_;
            }

            //- Return reference to the name of phi to allow adjustment
            word& phiName()
            {
                return phiName_;
            }

            //- Return the tangential velocity
            const vectorgpuField& tangentialVelocity() const
            {
                return tangentialVelocity_;
            }

            //- Reset the tangential velocity
            void setTangentialVelocity(const vectorgpuField& tangentialVelocity);


        // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchgpuFieldMapper&
            );

            //- Reverse map the given fvPatchField onto this fvPatchField
            virtual void rmap
            (
                const fvPatchVectorgpuField&,
                const labelgpuList&
            );


        //- Update the coefficients associated with the patch field
        virtual void updateCoeffs();

        //- Write
        virtual void write(Ostream&) const;


    // Member operators

        virtual void operator=(const fvPatchgpuField<vector>& pvf);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
