/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "pressureInletOutletVelocityFvPatchVectorgpuField.H"
#include "addToRunTimeSelectionTable.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
pressureInletOutletVelocityFvPatchVectorgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF
)
:
    directionMixedFvPatchVectorgpuField(p, iF),
    phiName_("phi")
{
    refValue() = Zero;
    refGrad() = Zero;
    valueFraction() = Zero;
}


Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
pressureInletOutletVelocityFvPatchVectorgpuField
(
    const pressureInletOutletVelocityFvPatchVectorgpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    directionMixedFvPatchVectorgpuField(ptf, p, iF, mapper),
    phiName_(ptf.phiName_)
{
    if (ptf.tangentialVelocity_.size())
    {
        tangentialVelocity_ = mapper(ptf.tangentialVelocity_);
    }
}


Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
pressureInletOutletVelocityFvPatchVectorgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    directionMixedFvPatchVectorgpuField(p, iF),
    phiName_(dict.getOrDefault<word>("phi", "phi"))
{
    patchType() = dict.getOrDefault<word>("patchType", word::null);
    fvPatchVectorgpuField::operator=(vectorgpuField("value", dict, p.size()));

    if (dict.found("tangentialVelocity"))
    {
        setTangentialVelocity
        (
            vectorgpuField("tangentialVelocity", dict, p.size())
        );
    }
    else
    {
        refValue() = Zero;
    }

    refGrad() = Zero;
    valueFraction() = Zero;
}


Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
pressureInletOutletVelocityFvPatchVectorgpuField
(
    const pressureInletOutletVelocityFvPatchVectorgpuField& pivpvf
)
:
    directionMixedFvPatchVectorgpuField(pivpvf),
    phiName_(pivpvf.phiName_),
    tangentialVelocity_(pivpvf.tangentialVelocity_)
{}


Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
pressureInletOutletVelocityFvPatchVectorgpuField
(
    const pressureInletOutletVelocityFvPatchVectorgpuField& pivpvf,
    const DimensionedgpuField<vector, gpuvolMesh>& iF
)
:
    directionMixedFvPatchVectorgpuField(pivpvf, iF),
    phiName_(pivpvf.phiName_),
    tangentialVelocity_(pivpvf.tangentialVelocity_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::
setTangentialVelocity(const vectorgpuField& tangentialVelocity)
{
    tangentialVelocity_ = tangentialVelocity;
    const vectorgpuField n(patch().nf());
    refValue() = tangentialVelocity_ - n*(n & tangentialVelocity_);
}


void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    directionMixedFvPatchVectorgpuField::autoMap(m);
    if (tangentialVelocity_.size())
    {
        tangentialVelocity_.autoMap(m);
    }
}


void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::rmap
(
    const fvPatchVectorgpuField& ptf,
    const labelgpuList& addr
)
{
    directionMixedFvPatchVectorgpuField::rmap(ptf, addr);

    if (tangentialVelocity_.size())
    {
        const pressureInletOutletVelocityFvPatchVectorgpuField& tiptf =
            refCast<const pressureInletOutletVelocityFvPatchVectorgpuField>(ptf);

        tangentialVelocity_.rmap(tiptf.tangentialVelocity_, addr);
    }
}


void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }

    const fvsPatchgpuField<scalar>& phip =
        patch().lookupPatchField<surfaceScalargpuField, scalar>(phiName_);

    valueFraction() = neg(phip)*(I - sqr(patch().nf()));

    directionMixedFvPatchVectorgpuField::updateCoeffs();
    directionMixedFvPatchVectorgpuField::evaluate();
}


void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::write
(
    Ostream& os
)
const
{
    fvPatchVectorgpuField::write(os);
    os.writeEntryIfDifferent<word>("phi", "phi", phiName_);
    if (tangentialVelocity_.size())
    {
        tangentialVelocity_.writeEntry("tangentialVelocity", os);
    }
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

void Foam::pressureInletOutletVelocityFvPatchVectorgpuField::operator=
(
    const fvPatchgpuField<vector>& pvf
)
{
    tmp<vectorgpuField> normalValue = transform(valueFraction(), refValue());
    tmp<vectorgpuField> transformGradValue = transform(I - valueFraction(), pvf);
    fvPatchgpuField<vector>::operator=(normalValue + transformGradValue);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchVectorgpuField,
        pressureInletOutletVelocityFvPatchVectorgpuField
    );
}

// ************************************************************************* //
