/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2016 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "outletPhaseMeanVelocityFvPatchVectorgpuField.H"
#include "volgpuFields.H"
#include "addToRunTimeSelectionTable.H"
#include "fvPatchgpuFieldMapper.H"
#include "surfacegpuFields.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::outletPhaseMeanVelocityFvPatchVectorgpuField
::outletPhaseMeanVelocityFvPatchVectorgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF
)
:
    mixedFvPatchgpuField<vector>(p, iF),
    Umean_(0),
    alphaName_("none")
{
    refValue() = Zero;
    refGrad() = Zero;
    valueFraction() = 0.0;
}


Foam::outletPhaseMeanVelocityFvPatchVectorgpuField
::outletPhaseMeanVelocityFvPatchVectorgpuField
(
    const outletPhaseMeanVelocityFvPatchVectorgpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    mixedFvPatchgpuField<vector>(ptf, p, iF, mapper),
    Umean_(ptf.Umean_),
    alphaName_(ptf.alphaName_)
{}


Foam::outletPhaseMeanVelocityFvPatchVectorgpuField
::outletPhaseMeanVelocityFvPatchVectorgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<vector, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    mixedFvPatchgpuField<vector>(p, iF),
    Umean_(dict.get<scalar>("Umean")),
    alphaName_(dict.lookup("alpha"))
{
    patchType() = dict.getOrDefault<word>("patchType", word::null);

    refValue() = Zero;
    refGrad() = Zero;
    valueFraction() = 0.0;

    if (dict.found("value"))
    {
        fvPatchVectorgpuField::operator=
        (
            vectorgpuField("value", dict, p.size())
        );
    }
    else
    {
        fvPatchVectorgpuField::operator=(patchInternalField());
    }
}


Foam::outletPhaseMeanVelocityFvPatchVectorgpuField
::outletPhaseMeanVelocityFvPatchVectorgpuField
(
    const outletPhaseMeanVelocityFvPatchVectorgpuField& ptf
)
:
    mixedFvPatchgpuField<vector>(ptf),
    Umean_(ptf.Umean_),
    alphaName_(ptf.alphaName_)
{}


Foam::outletPhaseMeanVelocityFvPatchVectorgpuField
::outletPhaseMeanVelocityFvPatchVectorgpuField
(
    const outletPhaseMeanVelocityFvPatchVectorgpuField& ptf,
    const DimensionedgpuField<vector, gpuvolMesh>& iF
)
:
    mixedFvPatchgpuField<vector>(ptf, iF),
    Umean_(ptf.Umean_),
    alphaName_(ptf.alphaName_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::outletPhaseMeanVelocityFvPatchVectorgpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }

    scalargpuField alphap =
        patch().lookupPatchField<volScalargpuField, scalar>(alphaName_);

    alphap = max(alphap, scalar(0));
    alphap = min(alphap, scalar(1));

    // Get the patchInternalField (zero-gradient field)
    vectorgpuField Uzg(patchInternalField());

    // Calculate the phase mean zero-gradient velocity
    scalar Uzgmean =
        gSum(alphap*(patch().Sf() & Uzg))
       /gSum(alphap*patch().magSf());

    // Set the refValue and valueFraction to adjust the boundary field
    // such that the phase mean is Umean_
    if (Uzgmean >= Umean_)
    {
        refValue() = Zero;
        valueFraction() = 1.0 - Umean_/Uzgmean;
    }
    else
    {
        refValue() = (Umean_ + Uzgmean)*patch().nf();
        valueFraction() = 1.0 - Uzgmean/Umean_;
    }

    mixedFvPatchgpuField<vector>::updateCoeffs();
}


void Foam::outletPhaseMeanVelocityFvPatchVectorgpuField::write
(
    Ostream& os
) const
{
    fvPatchgpuField<vector>::write(os);

    os.writeEntry("Umean", Umean_);
    os.writeEntry("alpha", alphaName_);
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
   makePatchTypeField
   (
       fvPatchVectorgpuField,
       outletPhaseMeanVelocityFvPatchVectorgpuField
   );
}


// ************************************************************************* //
