/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::inletOutletFvPatchField

Group
    grpOutletBoundaryConditions

Description
    This boundary condition provides a generic outflow condition, with
    specified inflow for the case of return flow.

Usage
    \table
        Property     | Description             | Required    | Default value
        phi          | Flux field name         | no          | phi
        inletValue   | Inlet value for reverse flow | yes    |
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            inletOutlet;
        phi             phi;
        inletValue      uniform 0;
        value           uniform 0;
    }
    \endverbatim

    The mode of operation is determined by the sign of the flux across the
    patch faces.

Note
    Sign conventions:
    - Positive flux (out of domain): apply zero-gradient condition
    - Negative flux (into of domain): apply the "inletValue" fixed-value

See also
    Foam::mixedFvPatchField
    Foam::zeroGradientFvPatchField
    Foam::outletInletFvPatchField

SourceFiles
    inletOutletFvPatchField.C

\*---------------------------------------------------------------------------*/

#ifndef inletOutletFvPatchgpuField_H
#define inletOutletFvPatchgpuField_H

#include "mixedFvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                   Class inletOutletFvPatchField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class inletOutletFvPatchgpuField
:
    public mixedFvPatchgpuField<Type>
{

protected:

    // Protected data

        //- Name of flux field
        word phiName_;


public:

    //- Runtime type information
    TypeName("inletOutlet");


    // Constructors

        //- Construct from patch and internal field
        inletOutletFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        inletOutletFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given inletOutletFvPatchField onto a new patch
        inletOutletFvPatchgpuField
        (
            const inletOutletFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        inletOutletFvPatchgpuField
        (
            const inletOutletFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new inletOutletFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        inletOutletFvPatchgpuField
        (
            const inletOutletFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new inletOutletFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Attributes

            //- Return true: this patch field is altered by assignment
            virtual bool assignable() const
            {
                return true;
            }


        //- Update the coefficients associated with the patch field
        virtual void updateCoeffs();

        //- Write
        virtual void write(Ostream&) const;


    // Member operators

        virtual void operator=(const fvPatchgpuField<Type>& pvf);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "inletOutletFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
