/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2017 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "fixedJumpAMIFvPatchgpuField.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::fixedJumpAMIFvPatchgpuField<Type>::fixedJumpAMIFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    jumpCyclicAMIFvPatchgpuField<Type>(p, iF),
    jump_(this->size(), Zero)
{}


template<class Type>
Foam::fixedJumpAMIFvPatchgpuField<Type>::fixedJumpAMIFvPatchgpuField
(
    const fixedJumpAMIFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    jumpCyclicAMIFvPatchgpuField<Type>(ptf, p, iF, mapper),
    jump_(ptf.jump_, mapper)
{}


template<class Type>
Foam::fixedJumpAMIFvPatchgpuField<Type>::fixedJumpAMIFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    jumpCyclicAMIFvPatchgpuField<Type>(p, iF),
    jump_(p.size(), Zero)
{
    if (this->cyclicAMIPatch().owner())
    {
        jump_ = gpuField<Type>("jump", dict, p.size());
    }

    if (dict.found("value"))
    {
        fvPatchgpuField<Type>::operator=
        (
            gpuField<Type>("value", dict, p.size())
        );
    }
    else
    {
        this->evaluate(Pstream::commsTypes::blocking);
    }
}


template<class Type>
Foam::fixedJumpAMIFvPatchgpuField<Type>::fixedJumpAMIFvPatchgpuField
(
    const fixedJumpAMIFvPatchgpuField<Type>& ptf
)
:
    jumpCyclicAMIFvPatchgpuField<Type>(ptf),
    jump_(ptf.jump_)
{}


template<class Type>
Foam::fixedJumpAMIFvPatchgpuField<Type>::fixedJumpAMIFvPatchgpuField
(
    const fixedJumpAMIFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    jumpCyclicAMIFvPatchgpuField<Type>(ptf, iF),
    jump_(ptf.jump_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::fixedJumpAMIFvPatchgpuField<Type>::jump() const
{
    if (this->cyclicAMIPatch().owner())
    {
        return jump_;
    }
    else
    {
        const fixedJumpAMIFvPatchgpuField& nbrPatch =
            refCast<const fixedJumpAMIFvPatchgpuField<Type>>
            (
                this->neighbourPatchField()
            );

        if (this->cyclicAMIPatch().applyLowWeightCorrection())
        {
            return this->cyclicAMIPatch().interpolate
            (
                nbrPatch.jump(),
                gpuField<Type>(this->size(), Zero)
            );
        }
        else
        {
            return this->cyclicAMIPatch().interpolate(nbrPatch.jump());
        }
    }
}


template<class Type>
void Foam::fixedJumpAMIFvPatchgpuField<Type>::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    jumpCyclicAMIFvPatchgpuField<Type>::autoMap(m);
    jump_.autoMap(m);
}


template<class Type>
void Foam::fixedJumpAMIFvPatchgpuField<Type>::rmap
(
    const fvPatchgpuField<Type>& ptf,
    const labelgpuList& addr
)
{
    jumpCyclicAMIFvPatchgpuField<Type>::rmap(ptf, addr);

    const fixedJumpAMIFvPatchgpuField<Type>& tiptf =
        refCast<const fixedJumpAMIFvPatchgpuField<Type>>(ptf);
    jump_.rmap(tiptf.jump_, addr);
}


template<class Type>
void Foam::fixedJumpAMIFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    os.writeEntry("patchType", this->interfaceFieldType());

    if (this->cyclicAMIPatch().owner())
    {
        jump_.writeEntry("jump", os);
    }

    this->writeEntry("value", os);
}


// ************************************************************************* //
