/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicAMIFvPatchgpuField

Group
    grpCoupledBoundaryConditions

Description
    This boundary condition enforces a cyclic condition between a pair of
    boundaries, whereby communication between the patches is performed using
    an arbitrary mesh interface (AMI) interpolation.

Usage
    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            cyclicAMI;
    }
    \endverbatim

Note
    The outer boundary of the patch pairs must be similar, i.e. if the owner
    patch is transformed to the neighbour patch, the outer perimiter of each
    patch should be identical (or very similar).

See also
    Foam::AMIInterpolation

SourceFiles
    cyclicAMIFvPatchgpuField.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicAMIFvPatchgpuField_H
#define cyclicAMIFvPatchgpuField_H


#include "coupledFvPatchgpuField.H"
#include "cyclicAMILduInterfacegpuField.H"
#include "cyclicAMIgpuFvPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                    Class cyclicAMIFvPatchgpuField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class cyclicAMIFvPatchgpuField
:
    virtual public cyclicAMILduInterfacegpuField,
    public coupledFvPatchgpuField<Type>
{
    // Private data

        //- Local reference cast into the cyclic patch
        const cyclicAMIgpuFvPatch& cyclicAMIPatch_;


    // Private Member Functions

        //- Return neighbour side field given internal fields
        template<class Type2>
        tmp<gpuField<Type2>> neighbourSideField
        (
            const gpuField<Type2>&
        ) const;


        //- Return new matrix coeffs
        tmp<gpuField<scalar>> coeffs
        (
            gpufvMatrix<Type>& matrix,
            const gpuField<scalar>&,
            const label
        ) const;


        template<class Type2>
        void collectStencilData
        (
            const refPtr<mapDistribute>& mapPtr,
            const labelListList& stencil,
            const Type2& data,
            List<Type2>& expandedData
        );


public:

    //- Runtime type information
    TypeName(cyclicAMIgpuFvPatch::typeName_());


    // Constructors

        //- Construct from patch and internal field
        cyclicAMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        cyclicAMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given cyclicAMIFvPatchgpuField onto a new patch
        cyclicAMIFvPatchgpuField
        (
            const cyclicAMIFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        cyclicAMIFvPatchgpuField(const cyclicAMIFvPatchgpuField<Type>&);

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicAMIFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        cyclicAMIFvPatchgpuField
        (
            const cyclicAMIFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicAMIFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Access

            //- Return local reference cast into the cyclic AMI patch
            const cyclicAMIgpuFvPatch& cyclicAMIPatch() const
            {
                return cyclicAMIPatch_;
            }


        // Evaluation functions

            //- Return true if coupled. Note that the underlying patch
            //  is not coupled() - the points don't align.
            virtual bool coupled() const;

            //- Return neighbour coupled internal cell data
            virtual tmp<gpuField<Type>> patchNeighbourField() const;

            //- Return reference to neighbour patchField
            const cyclicAMIFvPatchgpuField<Type>& neighbourPatchField() const;

            //- Update result field based on interface functionality

            virtual void updateInterfaceMatrix
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const;

            //- Update result field based on interface functionality
            virtual void updateInterfaceMatrix
            (
                gpuField<Type>&,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const gpuField<Type>&,
                const scalargpuField&,
                const Pstream::commsTypes commsType
            ) const;


            //- Manipulate matrix
            virtual void manipulateMatrix
            (
                gpufvMatrix<Type>& m,
                const label iMatrix,
                const direction cmpt
            );


        // Cyclic AMI coupled interface functions

            //- Does the patch field perform the transformation
            virtual bool doTransform() const
            {
                return
                    !(cyclicAMIPatch_.parallel() || pTraits<Type>::rank == 0);
            }

            //- Return face transformation tensor

            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicAMIPatch_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor

            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicAMIPatch_.gpuReverseT();
            }

            //- Return rank of component for transform
            virtual int rank() const
            {
                return pTraits<Type>::rank;
            }


        // I-O

            //- Write
            virtual void write(Ostream& os) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "cyclicAMIFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
