/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicFvPatchgpuField

Group
    grpCoupledBoundaryConditions

Description
    This boundary condition enforces a cyclic condition between a pair of
    boundaries.

Usage
    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            cyclic;
    }
    \endverbatim

Note
    The patches must be topologically similar, i.e. if the owner patch is
    transformed to the neighbour patch, the patches should be identical (or
    very similar).

SourceFiles
    cyclicFvPatchgpuField.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicFvPatchgpuField_H
#define cyclicFvPatchgpuField_H

#include "coupledFvPatchgpuField.H"
#include "cyclicLduInterfacegpuField.H"
#include "cyclicgpuFvPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                        Class cyclicFvPatch Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class cyclicFvPatchgpuField
:
    virtual public cyclicLduInterfacegpuField,
    public coupledFvPatchgpuField<Type>
{
    // Private data

        //- Local reference cast into the cyclic patch
        const cyclicgpuFvPatch& cyclicPatch_;


    // Private Member Functions

        //- Return neighbour side field given internal fields
        template<class Type2>
        tmp<gpuField<Type2>> neighbourSideField
        (
            const gpuField<Type2>&
        ) const;


public:


    //- Runtime type information
    TypeName(cyclicgpuFvPatch::typeName_());


    // Constructors

        //- Construct from patch and internal field
        cyclicFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        cyclicFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&,
            const bool valueRequired = true
        );

        //- Construct by mapping given cyclicFvPatchgpuField onto a new patch
        cyclicFvPatchgpuField
        (
            const cyclicFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        cyclicFvPatchgpuField
        (
            const cyclicFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        cyclicFvPatchgpuField
        (
            const cyclicFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Access

            //- Return local reference cast into the cyclic patch
            const cyclicgpuFvPatch& cyclicPatch() const
            {
                return cyclicPatch_;
            }


        // Evaluation functions

            //- Return neighbour coupled internal cell data
            tmp<gpuField<Type>> patchNeighbourField() const;

            //- Return reference to neighbour patchField
            const cyclicFvPatchgpuField<Type>& neighbourPatchField() const;

            //- Update result field based on interface functionality

            virtual void updateInterfaceMatrix
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const;

            virtual void updateInterfaceMatrix
            (
                gpuField<Type>&,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const gpuField<Type>&,
                const scalargpuField&,
                const Pstream::commsTypes commsType
            ) const;
            //- Update result field based on interface functionality


            //- Manipulate matrix
            virtual void manipulateMatrix
            (
                gpufvMatrix<Type>& m,
                const label iMatrix,
                const direction cmp
            );



        // Cyclic coupled interface functions

            //- Does the patch field perform the transformation
            virtual bool doTransform() const
            {
                return !(cyclicPatch_.parallel() || pTraits<Type>::rank == 0);
            }

            //- Return face transformation tensor
            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicPatch_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor
            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicPatch_.gpuReverseT();
            }

            //- Return rank of component for transform
            virtual int rank() const
            {
                return pTraits<Type>::rank;
            }


        // I-O

            //- Write
            virtual void write(Ostream& os) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "cyclicFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
