/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "fv.H"
#include "objectRegistry.H"
#include "solution.H"

// * * * * * * * * * * * * * * * * * Selectors * * * * * * * * * * * * * * * //

template<class Type>
Foam::tmp<Foam::fv::gpugradScheme<Type>> Foam::fv::gpugradScheme<Type>::New
(
    const gpufvMesh& mesh,
    Istream& schemeData
)
{
    if (fv::debug)
    {
        InfoInFunction << "Constructing gradScheme<Type>" << endl;
    }

    if (schemeData.eof())
    {
        FatalIOErrorInFunction(schemeData)
            << "Grad scheme not specified" << endl << endl
            << "Valid grad schemes are :" << endl
            << IstreamConstructorTablePtr_->sortedToc()
            << exit(FatalIOError);
    }

    const word schemeName(schemeData);

    auto* ctorPtr = IstreamConstructorTable(schemeName);

    if (!ctorPtr)
    {
        FatalIOErrorInLookup
        (
            schemeData,
            "gpugrad",
            schemeName,
            *IstreamConstructorTablePtr_
        ) << exit(FatalIOError);
    }

    return ctorPtr(mesh, schemeData);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type>
Foam::tmp
<
    Foam::GeometricgpuField
    <
        typename Foam::outerProduct<Foam::vector, Type>::type,
        Foam::fvPatchgpuField,
        Foam::gpuvolMesh
    >
>
Foam::fv::gpugradScheme<Type>::grad
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vsf,
    const word& name
) const
{
    typedef typename outerProduct<vector, Type>::type GradType;
    typedef GeometricgpuField<GradType, fvPatchgpuField, gpuvolMesh> GradFieldType;

    GradFieldType* pgGrad =
        mesh().hostmesh().objectRegistry::template getObjectPtr<GradFieldType>(name);

    if (!this->mesh().hostmesh().cache(name) || this->mesh().hostmesh().changing())
    {
        // Delete any old occurrences to avoid double registration
        if (pgGrad && pgGrad->ownedByRegistry())
        {
            solution::cachePrintMessage("Deleting", name, vsf);
            delete pgGrad;
        }

        solution::cachePrintMessage("Calculating", name, vsf);
        return calcGrad(vsf, name);
    }


    if (!pgGrad)
    {
        solution::cachePrintMessage("Calculating and caching", name, vsf);

        pgGrad = calcGrad(vsf, name).ptr();
        regIOobject::store(pgGrad);
    }
    else
    {
        if (pgGrad->upToDate(vsf))
        {
            solution::cachePrintMessage("Reusing", name, vsf);
        }
        else
        {
            solution::cachePrintMessage("Updating", name, vsf);
            delete pgGrad;

            pgGrad = calcGrad(vsf, name).ptr();
            regIOobject::store(pgGrad);
        }
    }

    return *pgGrad;
}


template<class Type>
Foam::tmp
<
    Foam::GeometricgpuField
    <
        typename Foam::outerProduct<Foam::vector, Type>::type,
        Foam::fvPatchgpuField,
        Foam::gpuvolMesh
    >
>
Foam::fv::gpugradScheme<Type>::grad
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vsf
) const
{
    return grad(vsf, "grad(" + vsf.name() + ')');
}


template<class Type>
Foam::tmp
<
    Foam::GeometricgpuField
    <
        typename Foam::outerProduct<Foam::vector, Type>::type,
        Foam::fvPatchgpuField,
        Foam::gpuvolMesh
    >
>
Foam::fv::gpugradScheme<Type>::grad
(
    const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvsf
) const
{
    typedef typename outerProduct<vector, Type>::type GradType;
    typedef GeometricgpuField<GradType, fvPatchgpuField, gpuvolMesh> GradFieldType;

    tmp<GradFieldType> tgrad = grad(tvsf());
    tvsf.clear();
    return tgrad;
}


// ************************************************************************* //
