/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvOptions.H"
#include "gpufvMesh.H"
#include "Time.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    namespace fv
    {
        defineTypeNameAndDebug(gpuoptions, 0);
    }
}


// * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * * //

Foam::IOobject Foam::fv::gpuoptions::createIOobject
(
    const gpufvMesh& mesh
) const
{
    IOobject io
    (
        typeName,
        mesh.time().constant(),
        mesh.hostmesh(),
        IOobject::MUST_READ,
        IOobject::NO_WRITE
    );

    if (io.typeHeaderOk<IOdictionary>(true))
    {
        Info<< "Creating finite volume options from "
            << io.instance()/io.name() << nl
            << endl;

        io.readOpt(IOobject::MUST_READ_IF_MODIFIED);
    }
    else
    {
        // Check if the fvOptions file is in system
        io.instance() = mesh.time().system();

        if (io.typeHeaderOk<IOdictionary>(true))
        {
            Info<< "Creating finite volume options from "
                << io.instance()/io.name() << nl
                << endl;

            io.readOpt(IOobject::MUST_READ_IF_MODIFIED);
        }
        else
        {
            io.readOpt(IOobject::NO_READ);
        }
    }

    return io;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::fv::gpuoptions::gpuoptions
(
    const gpufvMesh& mesh
)
:
    IOdictionary(createIOobject(mesh)),
    gpuoptionList(mesh, *this)
{}


Foam::fv::gpuoptions& Foam::fv::gpuoptions::New(const gpufvMesh& mesh)
{
    if (mesh.thisDb().foundObject<gpuoptions>(typeName))
    {
        return const_cast<gpuoptions&>
        (
            mesh.hostmesh().lookupObject<gpuoptions>(typeName)
        );
    }
    else
    {
        DebugInFunction
            << "Constructing " << typeName
            << " for region " << mesh.name() << nl;

        gpuoptions* objectPtr = new gpuoptions(mesh);
        regIOobject::store(objectPtr);
        return *objectPtr;
    }
}


bool Foam::fv::gpuoptions::read()
{
    if (IOdictionary::regIOobject::read())
    {
        gpuoptionList::read(*this);
        return true;
    }

    return false;
}


// ************************************************************************* //
