/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2017 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpuMRFZoneList.H"
#include "volgpuFields.H"
#include "fixedValueFvsPatchgpuFields.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::gpuMRFZoneList::gpuMRFZoneList
(
    const gpufvMesh& mesh,
    const dictionary& dict
)
:
    PtrList<gpuMRFZone>(),
    mesh_(mesh)
{
    reset(dict);

    active(true);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

bool Foam::gpuMRFZoneList::active(const bool warn) const
{
    bool a = false;
    forAll(*this, i)
    {
        a = a || this->operator[](i).active();
    }

    if (warn && this->size() && !a)
    {
        Info<< "    No MRF zones active" << endl;
    }

    return a;
}


void Foam::gpuMRFZoneList::reset(const dictionary& dict)
{
    label count = 0;
    for (const entry& dEntry : dict)
    {
        if (dEntry.isDict())
        {
            ++count;
        }
    }

    this->resize(count);

    count = 0;
    for (const entry& dEntry : dict)
    {
        if (dEntry.isDict())
        {
            const word& name = dEntry.keyword();
            const dictionary& modelDict = dEntry.dict();

            Info<< "    creating MRF zone: " << name << endl;

            this->set
            (
                count++,
                new gpuMRFZone(name, mesh_, modelDict)
            );
        }
    }
}


const Foam::gpuMRFZone& Foam::gpuMRFZoneList::getFromName
(
    const word& name
) const
{
    DynamicList<word> names;
    for (const auto& mrf: *this)
    {
        if (mrf.name() == name)
        {
            return mrf;
        }

        names.append(mrf.name());
    }

    FatalErrorInFunction
        << "Unable to find gupMRFZone " << name
        << ". Available zones are: " << names
        << exit(FatalError);

    return first();
}


bool Foam::gpuMRFZoneList::read(const dictionary& dict)
{
    bool allOk = true;
    for (auto& mrf: *this)
    {
        bool ok = mrf.read(dict.subDict(mrf.name()));
        allOk = (allOk && ok);
    }
    return allOk;
}


bool Foam::gpuMRFZoneList::writeData(Ostream& os) const
{
    for (const auto& mrf: *this)
    {
        os  << nl;
        mrf.writeData(os);
    }

    return os.good();
}


void Foam::gpuMRFZoneList::addAcceleration
(
    const volVectorgpuField& U,
    volVectorgpuField& ddtU
) const
{
    for (const auto& mrf: *this)
    {
        mrf.addCoriolis(U, ddtU);
    }
}


void Foam::gpuMRFZoneList::addAcceleration(gpufvVectorMatrix& UEqn) const
{
    for (const auto& mrf: *this)
    {
        mrf.addCoriolis(UEqn);
    }
}


void Foam::gpuMRFZoneList::addAcceleration
(
    const volScalargpuField& rho,
    gpufvVectorMatrix& UEqn
) const
{
    for (const auto& mrf: *this)
    {
        mrf.addCoriolis(rho, UEqn);
    }
}


Foam::tmp<Foam::volVectorgpuField> Foam::gpuMRFZoneList::DDt
(
    const volVectorgpuField& U
) const
{
    auto tacceleration =
        tmp<volVectorgpuField>::New
        (
            IOobject
            (
                "gpuMRFZoneList:acceleration",
                U.mesh().hostmesh().time().timeName(),
                U.mesh().hostmesh()
            ),
            U.mesh(),
            dimensionedVector(U.dimensions()/dimTime, Zero)
        );
    auto& acceleration = tacceleration.ref();

    for (const auto& mrf: *this)
    {
        mrf.addCoriolis(U, acceleration);
    }

    return tacceleration;
}


Foam::tmp<Foam::volVectorgpuField> Foam::gpuMRFZoneList::DDt
(
    const volScalargpuField& rho,
    const volVectorgpuField& U
) const
{
    return rho*DDt(U);
}


void Foam::gpuMRFZoneList::makeRelative(volVectorgpuField& U) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeRelative(U);
    }
}


void Foam::gpuMRFZoneList::makeRelative(surfaceScalargpuField& phi) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeRelative(phi);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::gpuMRFZoneList::relative
(
    const tmp<surfaceScalargpuField>& tphi
) const
{
    if (size())
    {
        tmp<surfaceScalargpuField> rphi
        (
            New
            (
                tphi,
                "relative(" + tphi().name() + ')',
                tphi().dimensions(),
                true
            )
        );

        makeRelative(rphi.ref());

        tphi.clear();

        return rphi;
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


Foam::tmp<Foam::FieldField<Foam::fvsPatchgpuField, Foam::scalar>>
Foam::gpuMRFZoneList::relative
(
    const tmp<FieldField<fvsPatchgpuField, scalar>>& tphi
) const
{
    if (size())
    {
        tmp<FieldField<fvsPatchgpuField, scalar>> rphi(New(tphi, true));

        for (const auto& mrf: *this)
        {
            mrf.makeRelative(rphi.ref());
        }

        tphi.clear();

        return rphi;
    }
    else
    {
        return tmp<FieldField<fvsPatchgpuField, scalar>>(tphi, true);
    }
}


Foam::tmp<Foam::gpuField<Foam::scalar>>
Foam::gpuMRFZoneList::relative
(
    const tmp<gpuField<scalar>>& tphi,
    const label patchi
) const
{
    if (size())
    {
        tmp<gpuField<scalar>> rphi(New(tphi, true));

        for (const auto& mrf: *this)
        {
            mrf.makeRelative(rphi.ref(), patchi);
        }

        tphi.clear();

        return rphi;
    }
    else
    {
        return tmp<gpuField<scalar>>(tphi, true);
    }
}


void Foam::gpuMRFZoneList::makeRelative
(
    const surfaceScalargpuField& rho,
    surfaceScalargpuField& phi
) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeRelative(rho, phi);
    }
}


void Foam::gpuMRFZoneList::makeAbsolute(volVectorgpuField& U) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeAbsolute(U);
    }
}


void Foam::gpuMRFZoneList::makeAbsolute(surfaceScalargpuField& phi) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeAbsolute(phi);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::gpuMRFZoneList::absolute
(
    const tmp<surfaceScalargpuField>& tphi
) const
{
    if (size())
    {
        tmp<surfaceScalargpuField> rphi
        (
            New
            (
                tphi,
                "absolute(" + tphi().name() + ')',
                tphi().dimensions(),
                true
            )
        );

        makeAbsolute(rphi.ref());

        tphi.clear();

        return rphi;
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


void Foam::gpuMRFZoneList::makeAbsolute
(
    const surfaceScalargpuField& rho,
    surfaceScalargpuField& phi
) const
{
    for (const auto& mrf: *this)
    {
        mrf.makeAbsolute(rho, phi);
    }
}


void Foam::gpuMRFZoneList::correctBoundaryVelocity(volVectorgpuField& U) const
{
    for (const auto& mrf: *this)
    {
        mrf.correctBoundaryVelocity(U);
    }
}


void Foam::gpuMRFZoneList::correctBoundaryFlux
(
    const volVectorgpuField& U,
    surfaceScalargpuField& phi
) const
{
    FieldField<fvsPatchgpuField, scalar> Uf
    (
        relative(mesh_.Sf().boundaryField() & U.boundaryField())
    );

    surfaceScalargpuField::Boundary& phibf = phi.boundaryFieldRef();

    forAll(mesh_.boundary(), patchi)
    {
        if (isA<fixedValueFvsPatchScalargpuField>(phibf[patchi]))
        {
            
            phibf[patchi] == Uf[patchi];
            
        }
    }
}


void Foam::gpuMRFZoneList::update()
{
    if (mesh_.hostmesh().topoChanging()) 
    {
        for (auto& mrf: *this)
        {
            mrf.update();
        }
    }
}


// * * * * * * * * * * * * * * * IOstream Operators  * * * * * * * * * * * * //

Foam::Ostream& Foam::operator<<
(
    Ostream& os,
    const gpuMRFZoneList& models
)
{
    models.writeData(os);
    return os;
}


// ************************************************************************* //
