/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2016-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GPBiCGStab.H"
#include "PrecisionAdaptor.H"

#include "lduMatrixSolverFunctors.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(GPBiCGStab, 0);

    gpulduMatrix::solver::addsymMatrixConstructorToTable<GPBiCGStab>
        addGPBiCGStabSymMatrixConstructorToTable_;

    gpulduMatrix::solver::addasymMatrixConstructorToTable<GPBiCGStab>
        addGPBiCGStabAsymMatrixConstructorToTable_;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::GPBiCGStab::GPBiCGStab
(
    const word& fieldName,
    const gpulduMatrix& matrix,
    const FieldField<gpuField, scalar>& interfaceBouCoeffs,
    const FieldField<gpuField, scalar>& interfaceIntCoeffs,
    const lduInterfacegpuFieldPtrsList& interfaces,
    const dictionary& solverControls
)
:
    gpulduMatrix::solver
    (
        fieldName,
        matrix,
        interfaceBouCoeffs,
        interfaceIntCoeffs,
        interfaces,
        solverControls
    )
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::solverPerformance Foam::GPBiCGStab::scalarSolve
(
    scalargpuField& psi,
    const scalargpuField& source,
    const direction cmpt
) const
{
    // --- Setup class containing solver performance data
    solverPerformance solverPerf
    (
        gpulduMatrix::preconditioner::getName(controlDict_) + typeName,
        fieldName_
    );

    const label nCells = psi.size();


    scalargpuField pA(nCells);

    scalargpuField yA(nCells);

    // --- Calculate A.psi
    matrix_.Amul(yA, psi, interfaceBouCoeffs_, interfaces_, cmpt);

    // --- Calculate initial residual field
    scalargpuField rA(source - yA);

    /*matrix().setResidualField
    (
        ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
        fieldName_,
        true
    );*/

    // --- Calculate normalisation factor
    const solveScalar normFactor = this->normFactor(psi, source, yA, pA);

    if ((log_ >= 2) || (gpulduMatrix::debug >= 2))
    {
        Info<< "   Normalisation factor = " << normFactor << endl;
    }

    // --- Calculate normalised residual norm
    solverPerf.initialResidual() =
        gSumMag(rA, matrix().mesh().comm())
       /normFactor;
    solverPerf.finalResidual() = solverPerf.initialResidual();

    // --- Check convergence, solve if not converged
    if
    (
        minIter_ > 0
     || !solverPerf.checkConvergence(tolerance_, relTol_, log_)
    )
    {
        scalargpuField AyA(nCells);

        scalargpuField sA(nCells);

        scalargpuField zA(nCells);

        scalargpuField tA(nCells);

        // --- Store initial residual
        const scalargpuField rA0(rA);

        // --- Initial values not used
        solveScalar rA0rA = 0;
        solveScalar alpha = 0;
        solveScalar omega = 0;

        // --- Select and construct the preconditioner
        autoPtr<gpulduMatrix::preconditioner> preconPtr =
        gpulduMatrix::preconditioner::New
        (
            *this,
            controlDict_
        );

        // --- Solver iteration
        do
        {
            // --- Store previous rA0rA
            const solveScalar rA0rAold = rA0rA;

            rA0rA = gSumProd(rA0, rA, matrix().mesh().comm());

            // --- Test for singularity
            if (solverPerf.checkSingularity(mag(rA0rA)))
            {
                break;
            }

            // --- Update pA
            if (solverPerf.nIterations() == 0)
            {
                thrust::copy(rA.begin(),rA.end(),pA.begin());
            }
            else
            {
                // --- Test for singularity
                if (solverPerf.checkSingularity(mag(omega)))
                {
                    break;
                }

                const solveScalar beta = (rA0rA/rA0rAold)*(alpha/omega);

                thrust::transform
                (
                    pA.begin(),
                    pA.end(),
                    AyA.begin(),
                    pA.begin(),
                    pAMinusOmegaAyAFunctor(omega)
                );

                thrust::transform
                (
                    rA.begin(),
                    rA.end(),
                    pA.begin(),
                    pA.begin(),
                    rAPlusBetaPAFunctor(beta)
                );
            }

            // --- Precondition pA
            preconPtr->precondition(yA, pA, cmpt);

            // --- Calculate AyA
            matrix_.Amul(AyA, yA, interfaceBouCoeffs_, interfaces_, cmpt);

            const solveScalar rA0AyA =
                gSumProd(rA0, AyA, matrix().mesh().comm());

            alpha = rA0rA/rA0AyA;

            // --- Calculate sA
            thrust::transform
            (
                rA.begin(),
                rA.end(),
                AyA.begin(),
                sA.begin(),
                rAMinusAlphaAyAFunctor(alpha)
            );

            // --- Test sA for convergence
            solverPerf.finalResidual() =
                gSumMag(sA, matrix().mesh().comm())/normFactor;

            if
            (
                solverPerf.nIterations() >= minIter_
             && solverPerf.checkConvergence(tolerance_, relTol_, log_)
            )
            {
                thrust::transform
            	(
                	psi.begin(),
                	psi.end(),
                	yA.begin(),
                	psi.begin(),
                	psiPlusAlphaYAFunctor(alpha)
            	);

                solverPerf.nIterations()++;

                return solverPerf;
            }

            // --- Precondition sA
            preconPtr->precondition(zA, sA, cmpt);

            // --- Calculate tA
            matrix_.Amul(tA, zA, interfaceBouCoeffs_, interfaces_, cmpt);

            const solveScalar tAtA = gSumSqr(tA, matrix().mesh().comm());

            // --- Calculate omega from tA and sA
            //     (cheaper than using zA with preconditioned tA)
            omega = gSumProd(tA, sA, matrix().mesh().comm())/tAtA;

            // --- Update solution and residual
            thrust::transform
            (
                psi.begin(),
                psi.end(),
                yA.begin(),
                psi.begin(),
                psiPlusAlphaYAFunctor(alpha)
            );

            thrust::transform
            (
                psi.begin(),
                psi.end(),
                zA.begin(),
                psi.begin(),
                psiPlusOmegaZAFunctor(omega)
            );

            thrust::transform
            (
                sA.begin(),
                sA.end(),
                tA.begin(),
                rA.begin(),
                sAMinusOmegaTAFunctor(omega)
            );

            solverPerf.finalResidual() =
                gSumMag(rA, matrix().mesh().comm())
               /normFactor;
        } while
        (
            (
              ++solverPerf.nIterations() < maxIter_
            && !solverPerf.checkConvergence(tolerance_, relTol_, log_)
            )
         || solverPerf.nIterations() < minIter_
        );
    }

    /*matrix().setResidualField
    (
        ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
        fieldName_,
        false
    );*/

    return solverPerf;
}


Foam::solverPerformance Foam::GPBiCGStab::solve
(
    scalargpuField& psi_s,
    const scalargpuField& source,
    const direction cmpt
) const
{
    return scalarSolve
    (
        psi_s,
        source,
        cmpt
    );
}

// ************************************************************************* //
