/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GGAMGAgglomeration.H"
#include "GGAMGInterface.H"
#include "processorGGAMGInterface.H"
#include "cyclicgpuLduInterface.H"

#include <thrust/iterator/discard_iterator.h>
#include <thrust/sort.h>
#include <thrust/scan.h>
#include <thrust/unique.h>

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::GGAMGAgglomeration::createSort
(
    const labelgpuList& list, 
    labelgpuList& sort
)
{
    sort.setSize(list.size());

    thrust::copy
    (
        thrust::make_counting_iterator(0),
        thrust::make_counting_iterator(0)+list.size(),
        sort.begin()
    );

    labelgpuList listTmp(list);
    thrust::stable_sort_by_key
    (
        listTmp.begin(),
        listTmp.end(),
        sort.begin()
    );
}

void Foam::GGAMGAgglomeration::createTarget
(
    const labelgpuList& list,
    const labelgpuList& sort,
    labelgpuList& target,
    labelgpuList& targetStart
)
{
    labelgpuList ones(list.size(),1);
    labelgpuList tmpSum(list.size());

    labelgpuList listSort(list.size());
    
    thrust::copy
    (
        thrust::make_permutation_iterator
        (
            list.begin(),
            sort.begin()
        ),
        thrust::make_permutation_iterator
        (
            list.begin(),
            sort.end()
        ),
        listSort.begin()
    );
 
    target = listSort;

    label targetSize = 
        thrust::unique
        (
            target.begin(),
            target.end()
        ) 
        - target.begin();
    target.setSize(targetSize);

    targetStart.setSize(list.size());
  
    thrust::reduce_by_key
    (
        listSort.begin(),
        listSort.end(),
        ones.begin(),
        thrust::make_discard_iterator(),
        tmpSum.begin()
    );

    thrust::exclusive_scan
    (
        tmpSum.begin(),
        tmpSum.end(),
        targetStart.begin()
    );

    targetStart.setSize(targetSize+1);
    targetStart.set(targetSize,list.size());
}


void Foam::GGAMGAgglomeration::buildFullRestrictAddr
(
    const labelgpuList& addr,
    const label level
)
{
    restrictSortAddressing_.set(level, new labelgpuField(addr.size()));
    restrictTargetAddressing_.set(level, new labelgpuField());
    restrictTargetStartAddressing_.set(level, new labelgpuField());

    createSort
    (
        addr,
        restrictSortAddressing_[level]
    );

    createTarget
    (
        addr,
        restrictSortAddressing_[level],
        restrictTargetAddressing_[level],
        restrictTargetStartAddressing_[level]
    );
}


void Foam::GGAMGAgglomeration::buildFullFaceRestrictAddr
(
    const labelgpuList& addr,
    const label level
)
{
    faceRestrictSortAddressing_.set(level, new labelgpuList(addr.size()));
    faceRestrictTargetAddressing_.set(level, new labelgpuList());
    faceRestrictTargetStartAddressing_.set(level, new labelgpuList());

    createSort
    (
        addr,
        faceRestrictSortAddressing_[level]
    );

    createTarget
    (
        addr,
        faceRestrictSortAddressing_[level],
        faceRestrictTargetAddressing_[level],
        faceRestrictTargetStartAddressing_[level]
    );
}


void Foam::GGAMGAgglomeration::buildFullPatchFaceRestrictAddr
(
    const labelgpuListList& addr,
    const label level
)
{
    patchFaceRestrictSortAddressing_.set
    ( 
        level,
        new labelgpuListList(addr.size())
    );
    patchFaceRestrictTargetAddressing_.set
    (
        level,
        new labelgpuListList(addr.size())
    );
    patchFaceRestrictTargetStartAddressing_.set
    ( 
        level,
        new labelgpuListList(addr.size())
    );

    labelgpuListList& patchFineToCoarseSortDevice =
        patchFaceRestrictSortAddressing_[level];
    labelgpuListList& patchFineToCoarseTargetDevice =
        patchFaceRestrictTargetAddressing_[level];
    labelgpuListList& patchFineToCoarseTargetStartDevice =
        patchFaceRestrictTargetStartAddressing_[level];

    forAll(addr,patchi)
    {
        createSort
        (
            addr[patchi],
            patchFineToCoarseSortDevice[patchi]
        );

        createTarget
        (
            addr[patchi],
            patchFineToCoarseSortDevice[patchi],
            patchFineToCoarseTargetDevice[patchi],
            patchFineToCoarseTargetStartDevice[patchi]
        );
    }
}


void Foam::GGAMGAgglomeration::buildFullAddressing()
{
    if(hasFullAddressing_) return;
    hasFullAddressing_ = true;

    forAll(restrictAddressing_, leveli)
    {
        buildFullRestrictAddr(restrictAddressing_[leveli], leveli);
    }

    forAll(faceRestrictAddressing_, leveli)
    {
        buildFullFaceRestrictAddr(faceRestrictAddressing_[leveli], leveli);
    }

    forAll(patchFaceRestrictAddressing_, leveli)
    {
        buildFullPatchFaceRestrictAddr(patchFaceRestrictAddressing_[leveli], leveli);
    }
}


void Foam::GGAMGAgglomeration::agglomerateLduAddressing
(
    const label fineLevelIndex
)
{
    const gpulduMesh& fineMesh = meshLevel(fineLevelIndex);
    const gpulduAddressing& fineMeshAddr = fineMesh.lduAddr();

    //const labelUList& upperAddr = fineMeshAddr.upperAddr();
    //const labelUList& lowerAddr = fineMeshAddr.lowerAddr();
    labelList lowerAddr(fineMeshAddr.lowerAddr().size());
    labelList upperAddr(fineMeshAddr.upperAddr().size());
    thrust::copy(fineMeshAddr.lowerAddr().begin(),fineMeshAddr.lowerAddr().end(),lowerAddr.begin());
    thrust::copy(fineMeshAddr.upperAddr().begin(),fineMeshAddr.upperAddr().end(),upperAddr.begin());

    const label nFineFaces = upperAddr.size();

    // Get restriction map for current level
    const labelField& restrictMap = restrictAddressingHost(fineLevelIndex);

    if (min(restrictMap) == -1)
    {
        FatalErrorInFunction
            << "min(restrictMap) == -1" << exit(FatalError);
    }

    if (restrictMap.size() != fineMeshAddr.size())
    {
        FatalErrorInFunction
            << "restrict map does not correspond to fine level. " << endl
            << " Sizes: restrictMap: " << restrictMap.size()
            << " nEqns: " << fineMeshAddr.size()
            << abort(FatalError);
    }


    // Get the number of coarse cells
    const label nCoarseCells = nCells_[fineLevelIndex];

    // Storage for coarse cell neighbours and coefficients

    // Guess initial maximum number of neighbours in coarse cell
    label maxNnbrs = 10;

    // Number of faces for each coarse-cell
    labelList cCellnFaces(nCoarseCells, Zero);

    // Setup initial packed storage for coarse-cell faces
    labelList cCellFaces(maxNnbrs*nCoarseCells);

    // Create face-restriction addressing
    faceRestrictAddressingHost_.set(fineLevelIndex, new labelList(nFineFaces));
    labelList& faceRestrictAddr = faceRestrictAddressingHost_[fineLevelIndex];

    // Initial neighbour array (not in upper-triangle order)
    labelList initCoarseNeighb(nFineFaces);

    // Counter for coarse faces
    label& nCoarseFaces = nFaces_[fineLevelIndex];
    nCoarseFaces = 0;

    // Loop through all fine faces
    forAll(upperAddr, fineFacei)
    {
        label rmUpperAddr = restrictMap[upperAddr[fineFacei]];
        label rmLowerAddr = restrictMap[lowerAddr[fineFacei]];

        if (rmUpperAddr == rmLowerAddr)
        {
            // For each fine face inside of a coarse cell keep the address
            // of the cell corresponding to the face in the faceRestrictAddr
            // as a negative index
            faceRestrictAddr[fineFacei] = -(rmUpperAddr + 1);
        }
        else
        {
            // this face is a part of a coarse face

            label cOwn = rmUpperAddr;
            label cNei = rmLowerAddr;

            // get coarse owner and neighbour
            if (rmUpperAddr > rmLowerAddr)
            {
                cOwn = rmLowerAddr;
                cNei = rmUpperAddr;
            }

            // check the neighbour to see if this face has already been found
            label* ccFaces = &cCellFaces[maxNnbrs*cOwn];

            bool nbrFound = false;
            label& ccnFaces = cCellnFaces[cOwn];

            for (int i=0; i<ccnFaces; i++)
            {
                if (initCoarseNeighb[ccFaces[i]] == cNei)
                {
                    nbrFound = true;
                    faceRestrictAddr[fineFacei] = ccFaces[i];
                    break;
                }
            }

            if (!nbrFound)
            {
                if (ccnFaces >= maxNnbrs)
                {
                    label oldMaxNnbrs = maxNnbrs;
                    maxNnbrs *= 2;

                    cCellFaces.setSize(maxNnbrs*nCoarseCells);

                    forAllReverse(cCellnFaces, i)
                    {
                        label* oldCcNbrs = &cCellFaces[oldMaxNnbrs*i];
                        label* newCcNbrs = &cCellFaces[maxNnbrs*i];

                        for (int j=0; j<cCellnFaces[i]; j++)
                        {
                            newCcNbrs[j] = oldCcNbrs[j];
                        }
                    }

                    ccFaces = &cCellFaces[maxNnbrs*cOwn];
                }

                ccFaces[ccnFaces] = nCoarseFaces;
                initCoarseNeighb[nCoarseFaces] = cNei;
                faceRestrictAddr[fineFacei] = nCoarseFaces;
                ccnFaces++;

                // new coarse face created
                nCoarseFaces++;
            }
        }
    } // end for all fine faces


    // Renumber into upper-triangular order

    // All coarse owner-neighbour storage
    labelList coarseOwner(nCoarseFaces);
    labelList coarseNeighbour(nCoarseFaces);
    labelList coarseFaceMap(nCoarseFaces);

    label coarseFacei = 0;

    forAll(cCellnFaces, cci)
    {
        label* cFaces = &cCellFaces[maxNnbrs*cci];
        label ccnFaces = cCellnFaces[cci];

        for (int i=0; i<ccnFaces; i++)
        {
            coarseOwner[coarseFacei] = cci;
            coarseNeighbour[coarseFacei] = initCoarseNeighb[cFaces[i]];
            coarseFaceMap[cFaces[i]] = coarseFacei;
            coarseFacei++;
        }
    }

    forAll(faceRestrictAddr, fineFacei)
    {
        if (faceRestrictAddr[fineFacei] >= 0)
        {
            faceRestrictAddr[fineFacei] =
                coarseFaceMap[faceRestrictAddr[fineFacei]];
        }
    }

    // Create face-flip status
    faceFlipMapHost_.set(fineLevelIndex, new boolList(nFineFaces, false));
    boolList& faceFlipMap = faceFlipMapHost_[fineLevelIndex];


    label nFlipped = 0;
    label nDissapear = 0;

    forAll(faceRestrictAddr, fineFacei)
    {
        label coarseFacei = faceRestrictAddr[fineFacei];

        if (coarseFacei >= 0)
        {
            // Maps to coarse face
            label cOwn = coarseOwner[coarseFacei];
            label cNei = coarseNeighbour[coarseFacei];

            label rmUpperAddr = restrictMap[upperAddr[fineFacei]];
            label rmLowerAddr = restrictMap[lowerAddr[fineFacei]];

            if (cOwn == rmUpperAddr && cNei == rmLowerAddr)
            {
                faceFlipMap[fineFacei] = true;
                nFlipped++;
            }
            else if (cOwn == rmLowerAddr && cNei == rmUpperAddr)
            {
                //faceFlipMap[fineFacei] = false;
            }
            else
            {
                FatalErrorInFunction
                    << "problem."
                    << " fineFacei:" << fineFacei
                    << " rmUpperAddr:" << rmUpperAddr
                    << " rmLowerAddr:" << rmLowerAddr
                    << " coarseFacei:" << coarseFacei
                    << " cOwn:" << cOwn
                    << " cNei:" << cNei
                    << exit(FatalError);
            }
        }
        else
        {
            nDissapear++;
        }
    }

    // Clear the temporary storage for the coarse cell data
    cCellnFaces.setSize(0);
    cCellFaces.setSize(0);
    initCoarseNeighb.setSize(0);
    coarseFaceMap.setSize(0);


    // Create coarse-level interfaces

    // Get reference to fine-level interfaces
    const gpulduInterfacePtrsList& fineInterfaces = interfaceLevel(fineLevelIndex);

    nPatchFaces_.set
    (
        fineLevelIndex,
        new labelList(fineInterfaces.size(), Zero)
    );
    labelList& nPatchFaces = nPatchFaces_[fineLevelIndex];

    patchFaceRestrictAddressingHost_.set
    (
        fineLevelIndex,
        new labelListList(fineInterfaces.size())
    );
    labelListList& patchFineToCoarse =
        patchFaceRestrictAddressingHost_[fineLevelIndex];


    const label nReq = Pstream::nRequests();

    // Initialise transfer of restrict addressing on the interface
    // The finest mesh uses patchAddr from the original lduAdressing.
    // the coarser levels create their own adressing for faceCells
    labelgpuField restrictMapDevice(restrictMap.size());
    thrust::copy(restrictMap.begin(),restrictMap.end(),restrictMapDevice.begin());

    forAll(fineInterfaces, inti)
    {
        if (fineInterfaces.set(inti))
        {
            if (fineLevelIndex == 0)
            {
                fineInterfaces[inti].initInternalFieldTransfer
                (
                    Pstream::commsTypes::nonBlocking,
                    restrictMapDevice,
                    fineMeshAddr.patchAddr(inti)
                );
            }
            else
            {
                fineInterfaces[inti].initInternalFieldTransfer
                (
                    Pstream::commsTypes::nonBlocking,
                    restrictMapDevice
                );
            }
        }
    }

    if (Pstream::parRun())
    {
        Pstream::waitRequests(nReq);
    }

labelgpuList coarseOwnerDevice(coarseOwner);
labelgpuList coarseNeighbourDevice(coarseNeighbour);
    // Add the coarse level
    meshLevels_.set
    (
        fineLevelIndex,
        new gpulduPrimitiveMesh
        (
            fineLevelIndex+1,
            nCoarseCells,
            coarseOwnerDevice,
            coarseNeighbourDevice,
            fineMesh.comm(),
            true
        )
    );

    gpulduInterfacePtrsList coarseInterfaces(fineInterfaces.size());

    forAll(fineInterfaces, inti)
    {
        if (fineInterfaces.set(inti))
        {
            labelField restrictMapInternalField;

            // The finest mesh uses patchAddr from the original lduAdressing.
            // the coarser levels create thei own adressing for faceCells
            if (fineLevelIndex == 0)
            {
                restrictMapInternalField = 
                    fineInterfaces[inti].interfaceInternalField
                    (
                        restrictMapDevice,
                        fineMeshAddr.patchAddr(inti)
                    )();
            }
            else
            {
                restrictMapInternalField = 
                    fineInterfaces[inti].interfaceInternalField
                    (
                        restrictMapDevice
                    )();
            }

            labelField nbrRestrictMapInternalField(
                fineInterfaces[inti].internalFieldTransfer
                (
                    Pstream::commsTypes::nonBlocking,
                    restrictMapDevice
                )());

            coarseInterfaces.set
            (
                inti,
                GGAMGInterface::New
                (
                    inti,
                    meshLevels_[fineLevelIndex].rawInterfaces(),
                    fineInterfaces[inti],
                    restrictMapInternalField,
                    nbrRestrictMapInternalField,
                    fineLevelIndex,
                    fineMesh.comm()
                ).ptr()
            );

            /* Same as below:
            coarseInterfaces.set
            (
                inti,
                GAMGInterface::New
                (
                    inti,
                    meshLevels_[fineLevelIndex].rawInterfaces(),
                    fineInterfaces[inti],
                    fineInterfaces[inti].interfaceInternalField(restrictMap),
                    fineInterfaces[inti].internalFieldTransfer
                    (
                        Pstream::commsTypes::nonBlocking,
                        restrictMap
                    ),
                    fineLevelIndex,
                    fineMesh.comm()
                ).ptr()
            );
            */

            nPatchFaces[inti] = coarseInterfaces[inti].gpuFaceCells().size();
            patchFineToCoarse[inti] = refCast<const GGAMGInterface>
            (
                coarseInterfaces[inti]
            ).faceRestrictAddressingHost();
        }
    }

    meshLevels_[fineLevelIndex].addInterfaces
    (
        coarseInterfaces,
        gpulduPrimitiveMesh::nonBlockingSchedule<processorGGAMGInterface>
        (
            coarseInterfaces
        )
    );

    //GPU addressing
    faceFlipMap_.set(fineLevelIndex, new boolgpuList(faceFlipMap));
    if (useAtomic())
    {
        faceRestrictAddressing_.set(fineLevelIndex, new labelgpuList(faceRestrictAddr));

        patchFaceRestrictSortAddressing_.set
        ( 
            fineLevelIndex,
            new labelgpuListList(patchFineToCoarse.size())
        );
        labelgpuListList& patchFaceRestrictAddressingTmp = 
            patchFaceRestrictSortAddressing_[fineLevelIndex];
        forAll(patchFineToCoarse,patchi)
        {
            patchFaceRestrictAddressingTmp[patchi] = patchFineToCoarse[patchi];
        }
    }
    else
    {
        labelgpuList faceRestrictAddressingTmp(faceRestrictAddr);
        buildFullFaceRestrictAddr(faceRestrictAddressingTmp, fineLevelIndex);

        labelgpuListList patchFaceRestrictAddressingTmp(patchFineToCoarse.size());
        forAll(patchFineToCoarse,patchi)
        {
            patchFaceRestrictAddressingTmp[patchi] = patchFineToCoarse[patchi];
        }
        buildFullPatchFaceRestrictAddr(patchFaceRestrictAddressingTmp, fineLevelIndex);
    }

    if (debug & 2)
    {
        Pout<< "GAMGAgglomeration :"
            << " agglomerated level " << fineLevelIndex
            << " from nCells:" << fineMeshAddr.size()
            << " nFaces:" << upperAddr.size()
            << " to nCells:" << nCoarseCells
            << " nFaces:" << nCoarseFaces
            << endl;
    }
}


/*void Foam::GAMGAgglomeration::procAgglomerateLduAddressing
(
    const label meshComm,
    const labelList& procAgglomMap,
    const labelList& procIDs,
    const label allMeshComm,

    const label levelIndex
)
{
    const lduMesh& myMesh = meshLevels_[levelIndex-1];


    procAgglomMap_.set(levelIndex, new labelList(procAgglomMap));
    agglomProcIDs_.set(levelIndex, new labelList(procIDs));
    procCommunicator_[levelIndex] = allMeshComm;

    // These could only be set on the master procs but it is
    // quite convenient to also have them on the slaves
    procCellOffsets_.set(levelIndex, new labelList(0));
    procFaceMap_.set(levelIndex, new labelListList(0));
    procBoundaryMap_.set(levelIndex, new labelListList(0));
    procBoundaryFaceMap_.set(levelIndex, new labelListListList(0));


    // Collect meshes
    PtrList<lduPrimitiveMesh> otherMeshes;
    lduPrimitiveMesh::gather(meshComm, myMesh, procIDs, otherMeshes);

    if (Pstream::myProcNo(meshComm) == procIDs[0])
    {
        // Combine all addressing

        labelList procFaceOffsets;
        meshLevels_.set
        (
            levelIndex-1,
            new lduPrimitiveMesh
            (
                allMeshComm,
                procAgglomMap,

                procIDs,
                myMesh,
                otherMeshes,

                procCellOffsets_[levelIndex],
                procFaceOffsets,
                procFaceMap_[levelIndex],
                procBoundaryMap_[levelIndex],
                procBoundaryFaceMap_[levelIndex]
            )
        );
    }


    // Combine restrict addressing
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~

    procAgglomerateRestrictAddressing
    (
        meshComm,
        procIDs,
        levelIndex
    );

    if (Pstream::myProcNo(meshComm) != procIDs[0])
    {
        clearLevel(levelIndex);
    }
}


void Foam::GAMGAgglomeration::procAgglomerateRestrictAddressing
(
    const label comm,
    const labelList& procIDs,
    const label levelIndex
)
{
    // Collect number of cells
    labelList nFineCells;
    gatherList
    (
        comm,
        procIDs,
        restrictAddressing_[levelIndex].size(),
        nFineCells
    );

    labelList offsets(nFineCells.size()+1);
    {
        offsets[0] = 0;
        forAll(nFineCells, i)
        {
            offsets[i+1] = offsets[i] + nFineCells[i];
        }
    }

    // Combine and renumber nCoarseCells
    labelList nCoarseCells;
    gatherList
    (
        comm,
        procIDs,
        nCells_[levelIndex],
        nCoarseCells
    );

    // (cell)restrictAddressing
    const globalIndex cellOffsetter(offsets);

    labelList procRestrictAddressing;
    cellOffsetter.gather
    (
        comm,
        procIDs,
        restrictAddressing_[levelIndex],
        procRestrictAddressing,

        UPstream::msgType(),
        Pstream::commsTypes::nonBlocking    //Pstream::commsTypes::scheduled
    );


    if (Pstream::myProcNo(comm) == procIDs[0])
    {
        labelList coarseCellOffsets(procIDs.size()+1);
        {
            coarseCellOffsets[0] = 0;
            forAll(procIDs, i)
            {
                coarseCellOffsets[i+1] = coarseCellOffsets[i]+nCoarseCells[i];
            }
        }

        nCells_[levelIndex] = coarseCellOffsets.last();

        // Renumber consecutively
        for (label proci = 1; proci < procIDs.size(); proci++)
        {
            SubList<label> procSlot
            (
                procRestrictAddressing,
                offsets[proci+1]-offsets[proci],
                offsets[proci]
            );
            forAll(procSlot, i)
            {
                procSlot[i] += coarseCellOffsets[proci];
            }
        }

        restrictAddressing_[levelIndex].transfer(procRestrictAddressing);
    }
}*/


void Foam::GGAMGAgglomeration::combineLevels(const label curLevel)
{
    label prevLevel = curLevel - 1;

    // Set the previous level nCells to the current
    nCells_[prevLevel] = nCells_[curLevel];
    nFaces_[prevLevel] = nFaces_[curLevel];

    // Map the restrictAddressing from the coarser level into the previous
    // finer level

    const labelList& curResAddr = restrictAddressingHost_[curLevel];
    labelList& prevResAddr = restrictAddressingHost_[prevLevel];

    const labelList& curFaceResAddr = faceRestrictAddressingHost_[curLevel];
    labelList& prevFaceResAddr = faceRestrictAddressingHost_[prevLevel];
    const boolList& curFaceFlipMap = faceFlipMapHost_[curLevel];
    boolList& prevFaceFlipMap = faceFlipMapHost_[prevLevel];

    forAll(prevFaceResAddr, i)
    {
        if (prevFaceResAddr[i] >= 0)
        {
            label fineFacei = prevFaceResAddr[i];
            prevFaceResAddr[i] = curFaceResAddr[fineFacei];
            prevFaceFlipMap[i] = curFaceFlipMap[fineFacei];
        }
        else
        {
            label fineFacei = -prevFaceResAddr[i] - 1;
            prevFaceResAddr[i] = -curResAddr[fineFacei] - 1;
            prevFaceFlipMap[i] = curFaceFlipMap[fineFacei];
        }
    }

    labelgpuList faceRestrictAddressingTmp(prevFaceResAddr);
    faceFlipMap_[prevLevel] = prevFaceFlipMap;
    faceRestrictSortAddressing_.set(prevLevel, new labelgpuList(prevFaceResAddr.size()));

    createSort
    (
        faceRestrictAddressingTmp,
        faceRestrictSortAddressing_[prevLevel]
    );

    createTarget
    (
        faceRestrictAddressingTmp,
        faceRestrictSortAddressing_[prevLevel],
        faceRestrictTargetAddressing_[prevLevel],
        faceRestrictTargetStartAddressing_[prevLevel]
    );
	
    // Delete the restrictAddressing for the coarser level
    faceRestrictAddressingHost_.set(curLevel, nullptr);
    faceFlipMapHost_.set(curLevel, nullptr);

    faceRestrictAddressing_.set(curLevel, nullptr);
    faceRestrictSortAddressing_.set(curLevel, nullptr);
    faceRestrictTargetAddressing_.set(curLevel, nullptr);
    faceRestrictTargetStartAddressing_.set(curLevel, nullptr);
    faceFlipMap_.set(curLevel, nullptr);

    forAll(prevResAddr, i)
    {
        prevResAddr[i] = curResAddr[prevResAddr[i]];
    }

    labelgpuList restrictAddressingTmp(prevResAddr);
    restrictSortAddressing_.set(prevLevel, new labelgpuField(prevResAddr.size()));

    createSort
    (
        restrictAddressingTmp,
        restrictSortAddressing_[prevLevel]
    );

    createTarget
    (
        restrictAddressingTmp,
        restrictSortAddressing_[prevLevel],
        restrictTargetAddressing_[prevLevel],
        restrictTargetStartAddressing_[prevLevel]
    );

    // Delete the restrictAddressing for the coarser level
    restrictAddressing_.set(curLevel, nullptr);
    restrictSortAddressing_.set(curLevel, nullptr);
    restrictTargetAddressing_.set(curLevel, nullptr);
    restrictTargetStartAddressing_.set(curLevel, nullptr);
    restrictAddressingHost_.set(curLevel, nullptr);

    const labelListList& curPatchFaceResAddrHost =
        patchFaceRestrictAddressingHost_[curLevel];
    labelListList& prevPatchFaceResAddrHost =
        patchFaceRestrictAddressingHost_[prevLevel];

    forAll(prevPatchFaceResAddrHost, inti)
    {
        const labelList& curResAddr = curPatchFaceResAddrHost[inti];
        labelList& prevResAddr = prevPatchFaceResAddrHost[inti];
        forAll(prevResAddr, i)
        {
            label fineFacei = prevResAddr[i];
            prevResAddr[i] = curResAddr[fineFacei];
        }

        labelgpuList patchFaceRestrictAddressingTmp(prevResAddr);

        createSort
        (
            patchFaceRestrictAddressingTmp,
            patchFaceRestrictSortAddressing_[prevLevel][inti]
        );

        createTarget
        (
            patchFaceRestrictAddressingTmp,
            patchFaceRestrictSortAddressing_[prevLevel][inti],
            patchFaceRestrictTargetAddressing_[prevLevel][inti],
            patchFaceRestrictTargetStartAddressing_[prevLevel][inti]
        );
    }

    // Patch faces
    nPatchFaces_[prevLevel] = nPatchFaces_[curLevel];

    // Adapt the restrict addressing for the patches
    const gpulduInterfacePtrsList& curInterLevel =
        meshLevels_[curLevel].rawInterfaces();
    const gpulduInterfacePtrsList& prevInterLevel =
        meshLevels_[prevLevel].rawInterfaces();

    forAll(prevInterLevel, inti)
    {
        if (prevInterLevel.set(inti))
        {
            GGAMGInterface& prevInt = refCast<GGAMGInterface>
            (
                const_cast<gpulduInterface&>
                (
                    prevInterLevel[inti]
                )
            );
            const GGAMGInterface& curInt = refCast<const GGAMGInterface>
            (
                curInterLevel[inti]
            );
            prevInt.combine(curInt);
        }
    }

    // Delete the matrix addressing and coefficients from the previous level
    // and replace with the corresponding entry from the coarser level
    meshLevels_.set(prevLevel, meshLevels_.set(curLevel, nullptr));
}


//void Foam::GAMGAgglomeration::gatherList
//(
//    const label comm,
//    const labelList& procIDs,
//
//    const label myVal,
//    labelList& vals,
//    const int tag
//)
//{
//    vals.setSize(procIDs.size());
//
//    if (Pstream::myProcNo(comm) == procIDs[0])
//    {
//        vals[0] = myVal;
//
//        for (label i=1; i<procIDs.size(); i++)
//        {
//            label& slaveVal = vals[i];
//            IPstream::read
//            (
//                Pstream::commsTypes::scheduled,
//                procIDs[i],
//                reinterpret_cast<char*>(&slaveVal),
//                sizeof(slaveVal),
//                tag,
//                comm
//            );
//        }
//    }
//    else
//    {
//        OPstream::write
//        (
//            Pstream::commsTypes::scheduled,
//            procIDs[0],
//            reinterpret_cast<const char*>(&myVal),
//            sizeof(myVal),
//            tag,
//            comm
//        );
//    }
//}


/*void Foam::GAMGAgglomeration::calculateRegionMaster
(
    const label comm,
    const labelList& procAgglomMap,
    labelList& masterProcs,
    List<label>& agglomProcIDs
)
{
    // Determine the master processors
    Map<label> agglomToMaster(procAgglomMap.size());

    forAll(procAgglomMap, proci)
    {
        const label coarsei = procAgglomMap[proci];

        auto iter = agglomToMaster.find(coarsei);
        if (iter.found())
        {
            iter.val() = min(iter.val(), proci);
        }
        else
        {
            agglomToMaster.insert(coarsei, proci);
        }
    }

    masterProcs.setSize(agglomToMaster.size());
    forAllConstIters(agglomToMaster, iter)
    {
        masterProcs[iter.key()] = iter.val();
    }


    // Collect all the processors in my agglomeration
    label myProcID = Pstream::myProcNo(comm);
    label myAgglom = procAgglomMap[myProcID];

    // Get all processors agglomerating to the same coarse
    // processor
    agglomProcIDs = findIndices(procAgglomMap, myAgglom);

    // Make sure the master is the first element.
    const label index =
        agglomProcIDs.find(agglomToMaster[myAgglom]);

    std::swap(agglomProcIDs[0], agglomProcIDs[index]);
}
*/

// ************************************************************************* //
