/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2016 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::laminarModels::Stokes

Group
    grpLaminar

Description
    Turbulence model for Stokes flow.

SourceFiles
    Stokes.C

\*---------------------------------------------------------------------------*/

#ifndef Stokes_H
#define Stokes_H

#include "laminarModel.H"
#include "linearViscousStress.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace laminarModels
{

/*---------------------------------------------------------------------------* \
                           Class Stokes Declaration
\*---------------------------------------------------------------------------*/

template<class BasicTurbulenceModel>
class Stokes
:
    public linearViscousStress<laminarModel<BasicTurbulenceModel>>
{

public:

    typedef typename BasicTurbulenceModel::alphaField alphaField;
    typedef typename BasicTurbulenceModel::rhoField rhoField;
    typedef typename BasicTurbulenceModel::transportModel transportModel;


    //- Runtime type information
    TypeName("Stokes");


    // Constructors

        //- Construct from components
        Stokes
        (
            const alphaField& alpha,
            const rhoField& rho,
            const volVectorgpuField& U,
            const surfaceScalargpuField& alphaRhoPhi,
            const surfaceScalargpuField& phi,
            const transportModel& transport,
            const word& propertiesName = turbulenceModel::propertiesName
        );


    // Selectors

        //- Return a reference to the selected turbulence model
        static autoPtr<Stokes> New
        (
            const alphaField& alpha,
            const rhoField& rho,
            const volVectorgpuField& U,
            const surfaceScalargpuField& alphaRhoPhi,
            const surfaceScalargpuField& phi,
            const transportModel& transport,
            const word& propertiesName = turbulenceModel::propertiesName
        );


    //- Destructor
    virtual ~Stokes() = default;


    // Member Functions

        //- Const access to the coefficients dictionary
        virtual const dictionary& coeffDict() const;

        //- Read turbulenceProperties dictionary
        virtual bool read();

        //- Return the turbulence viscosity, i.e. 0 for Stokes flow
        virtual tmp<volScalargpuField> nut() const;

        //- Return the turbulence viscosity on patch
        virtual tmp<scalargpuField> nut(const label patchi) const;

        //- Return the effective viscosity, i.e. the Stokes viscosity
        virtual tmp<volScalargpuField> nuEff() const;

        //- Return the effective viscosity on patch
        virtual tmp<scalargpuField> nuEff(const label patchi) const;

        //- Correct the Stokes viscosity
        virtual void correct();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace laminarModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "Stokes.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
