/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016, 2019 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "nutkRoughWallFunctionFvPatchScalargpuField.H"
#include "turbulenceModel.H"
#include "fvPatchgpuFieldMapper.H"
#include "volgpuFields.H"
#include "addToRunTimeSelectionTable.H"


// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //
namespace Foam
{

__host__ __device__
scalar fnRough
(
    const scalar KsPlus,
    const scalar Cs
)
{
    // Return fn based on non-dimensional roughness height

    if (KsPlus < 90.0)
    {
        return pow
        (
            (KsPlus - 2.25)/87.75 + Cs*KsPlus,
            sin(0.4258*(log(KsPlus) - 0.811))
        );
    }
    else
    {
        return (1.0 + Cs*KsPlus);
    }
}

struct nutkRWcalcNutFuntor
{
	const scalar EEE;
	const scalar Cmu25;
	const scalar kappa;

	nutkRWcalcNutFuntor
	(
		const scalar _EEE,
		const scalar _Cmu25,
		const scalar _kappa
	):
		EEE(_EEE),
		Cmu25(_Cmu25),
		kappa(_kappa)
	{}

	__host__ __device__
	scalar operator()(const thrust::tuple<scalar,scalar,scalar,scalar,scalar,scalar>& t)
	{
		const scalar k = thrust::get<0>(t);
		const scalar y = thrust::get<1>(t);
		const scalar nuw = thrust::get<2>(t);
		const scalar Ks = thrust::get<3>(t);
		const scalar Cs = thrust::get<4>(t);
		const scalar nutw = thrust::get<5>(t);

		const scalar uStar = Cmu25*sqrt(k);
        const scalar yPlus = uStar*y/nuw;
        const scalar KsPlus = uStar*Ks/nuw;
		
        scalar Edash = EEE;
        if (KsPlus < 2.25)
        {
            Edash /= fnRough(KsPlus, Cs);
        }
		
        scalar limitingNutw = max(nutw, nuw);
		
        scalar out =
            max
            (
                min
                (
                    nuw
                   *(yPlus*kappa/log(max(Edash*yPlus, 1+1e-4)) - 1),
                    2*limitingNutw
                ), 0.5*limitingNutw
            );

		return out;
	}
		
};
}




Foam::tmp<Foam::scalargpuField> Foam::nutkRoughWallFunctionFvPatchScalargpuField::
calcNut() const
{
    const label patchi = patch().index();

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );
    const scalargpuField& y = turbModel.y()[patchi];
    const tmp<volScalargpuField> tk = turbModel.k();
    const volScalargpuField& k = tk();
    const tmp<scalargpuField> tnuw = turbModel.nu(patchi);
    const scalargpuField& nuw = tnuw();

    const scalar Cmu25 = pow025(Cmu_);

    tmp<scalargpuField> tnutw(new scalargpuField(*this));
    scalargpuField& nutw = tnutw.ref();

	const labelgpuList& faceCell = patch().gpuFaceCells();

//    forAll(nutw, facei)
//    {
//        const label celli = patch().faceCells()[facei];
//
//        const scalar uStar = Cmu25*sqrt(k[celli]);
//        const scalar yPlus = uStar*y[facei]/nuw[facei];
//        const scalar KsPlus = uStar*Ks_[facei]/nuw[facei];
//
//        scalar Edash = E_;
//        if (2.25 < KsPlus)
//        {
//            Edash /= fnRough(KsPlus, Cs_[facei]);
//        }
//
//        const scalar limitingNutw = max(nutw[facei], nuw[facei]);
//
//        // To avoid oscillations limit the change in the wall viscosity
//        // which is particularly important if it temporarily becomes zero
//        nutw[facei] =
//            max
//            (
//                min
//                (
//                    nuw[facei]
//                   *(yPlus*kappa_/log(max(Edash*yPlus, 1+1e-4)) - 1),
//                    2*limitingNutw
//                ), 0.5*limitingNutw
//            );
//
//        if (debug)
//        {
//            Info<< "yPlus = " << yPlus
//                << ", KsPlus = " << KsPlus
//                << ", Edash = " << Edash
//                << ", nutw = " << nutw[facei]
//                << endl;
//        }
//    }
	thrust::transform
	(
		thrust::make_zip_iterator(thrust::make_tuple(
		thrust::make_permutation_iterator(k.begin(),faceCell.begin()),
		y.begin(),
		nuw.begin(),
		Ks_.begin(),
		Cs_.begin(),
		nutw.begin()
		)),
		thrust::make_zip_iterator(thrust::make_tuple(
		thrust::make_permutation_iterator(k.begin(),faceCell.end()),
		y.end(),
		nuw.end(),
		Ks_.end(),
		Cs_.end(),
		nutw.end()
		)),
		nutw.begin(),
		nutkRWcalcNutFuntor
		(
			E_,
			Cmu25,
			kappa_
		)
	);

    return tnutw;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::nutkRoughWallFunctionFvPatchScalargpuField::
nutkRoughWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    nutkWallFunctionFvPatchScalargpuField(p, iF),
    Ks_(p.size(), Zero),
    Cs_(p.size(), Zero)
{}


Foam::nutkRoughWallFunctionFvPatchScalargpuField::
nutkRoughWallFunctionFvPatchScalargpuField
(
    const nutkRoughWallFunctionFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    nutkWallFunctionFvPatchScalargpuField(ptf, p, iF, mapper),
    Ks_(ptf.Ks_, mapper),
    Cs_(ptf.Cs_, mapper)
{}


Foam::nutkRoughWallFunctionFvPatchScalargpuField::
nutkRoughWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    nutkWallFunctionFvPatchScalargpuField(p, iF, dict),
    Ks_("Ks", dict, p.size()),
    Cs_("Cs", dict, p.size())
{}


Foam::nutkRoughWallFunctionFvPatchScalargpuField::
nutkRoughWallFunctionFvPatchScalargpuField
(
    const nutkRoughWallFunctionFvPatchScalargpuField& rwfpsf
)
:
    nutkWallFunctionFvPatchScalargpuField(rwfpsf),
    Ks_(rwfpsf.Ks_),
    Cs_(rwfpsf.Cs_)
{}


Foam::nutkRoughWallFunctionFvPatchScalargpuField::
nutkRoughWallFunctionFvPatchScalargpuField
(
    const nutkRoughWallFunctionFvPatchScalargpuField& rwfpsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    nutkWallFunctionFvPatchScalargpuField(rwfpsf, iF),
    Ks_(rwfpsf.Ks_),
    Cs_(rwfpsf.Cs_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::nutkRoughWallFunctionFvPatchScalargpuField::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    nutkWallFunctionFvPatchScalargpuField::autoMap(m);
    Ks_.autoMap(m);
    Cs_.autoMap(m);
}


void Foam::nutkRoughWallFunctionFvPatchScalargpuField::rmap
(
    const fvPatchScalargpuField& ptf,
    const labelgpuList& addr
)
{
    nutkWallFunctionFvPatchScalargpuField::rmap(ptf, addr);

    const nutkRoughWallFunctionFvPatchScalargpuField& nrwfpsf =
        refCast<const nutkRoughWallFunctionFvPatchScalargpuField>(ptf);

    Ks_.rmap(nrwfpsf.Ks_, addr);
    Cs_.rmap(nrwfpsf.Cs_, addr);
}


void Foam::nutkRoughWallFunctionFvPatchScalargpuField::write
(
    Ostream& os
) const
{
    fvPatchgpuField<scalar>::write(os);
    writeLocalEntries(os);
    Cs_.writeEntry("Cs", os);
    Ks_.writeEntry("Ks", os);
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        nutkRoughWallFunctionFvPatchScalargpuField
    );
}

// ************************************************************************* //
