/*******************************************************************************************
 * This file contains the implementation of class functions for element and species.
 ******************************************************************************************/

#include "species.h"

/** Global variables for elements_d class types.
 */
elements_d *elements_d_ptr;

/** Global variables for species_d class types.
 */
species_d *species_d_ptr;

/** Construct class elements_d.
 *  
 *  \param[in] elements_const_d_ref  Pointer to device memory.
 */
elements_d::elements_d(elements_const *elements_const_ref) {

    int ele_num = elements_const_ref->ele_num;

    this->elements_const_h.ele_num = ele_num;
    unsigned int element_size = ele_num*sizeof(configItem);

    malloc_Host(this->elements_const_h.elements, element_size);

    if (elements_const_ref == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! WHEN SETTING ELEMENTS, Please initialize host pointer.\033[0m\n");
    } else {
        for (int i = 0; i < ele_num; i++) {
            strcpy(this->elements_const_h.elements[i].ele_names, elements_const_ref->elements[i].ele_names);
            this->elements_const_h.elements[i].ele_W = elements_const_ref->elements[i].ele_W;
        }
    }

#ifdef DEBUG
    configItem *elements = this->elements_const_h.elements;
    for (int i = 0; i < ele_num; i++)
        MPI_PRINTF("Element[%d]'s name is %s, and it's molar mass is %e\n", i, elements[i].ele_names, elements[i].ele_W);
#endif //DEBUG
}

/** Destory class elements_d.
 */
elements_d::~elements_d() {
    CUDACHECK(hipHostFree(this->elements_const_h.elements));
}

/** Construct class species_d.
 *  
 *  \param[in] species_const_d_ref  Pointer to device memory.
 */
species_d::species_d(species_const *species_const_ref) {

    int sp_num = species_const_ref->sp_num;

    this->species_const_d.sp_num = sp_num;

    if (species_const_ref == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! WHEN SETTING SPECIES, Please initialize host pointer.\033[0m\n");
    } else {
        DeviceDataset(this->species_const_d.sp_W,    species_const_ref->sp_W,       sp_num*sizeof(REAL));

        DeviceDataset(this->species_const_d.sp_name, species_const_ref->sp_name, 100*sp_num*sizeof(char));
        
        DeviceDataset(this->species_const_d.sp_nasa, species_const_ref->sp_nasa, 14*sp_num*sizeof(REAL));
        
        DeviceDataset(this->species_const_d.T_range, species_const_ref->T_range,  3*sp_num*sizeof(REAL));
    }

#ifdef DEBUG
    species_const species_const_c;

    malloc_Host(species_const_c.sp_W,       sp_num*sizeof(REAL));
    malloc_Host(species_const_c.sp_nasa, 14*sp_num*sizeof(REAL));
    malloc_Host(species_const_c.T_range,  3*sp_num*sizeof(REAL));

    DeviceDataget(this->species_const_d.sp_W,    species_const_c.sp_W,       sp_num*sizeof(REAL));
    DeviceDataget(this->species_const_d.sp_nasa, species_const_c.sp_nasa, 14*sp_num*sizeof(REAL));
    DeviceDataget(this->species_const_d.T_range, species_const_c.T_range,  3*sp_num*sizeof(REAL));

    MPI_PRINTF("\n Species's sp_w are: \n");
    for (int i = 0; i < sp_num; i++) {
        MPI_PRINTF("%e\t", species_const_c.sp_W[i]);
    }
    for (int i = 0; i < sp_num; i++) {
        MPI_PRINTF("\n Specie[%d]'s sp_nasa are: \n", i);
        for (int j = 0; j < 14; j++)
            MPI_PRINTF("%e\t", species_const_c.sp_nasa[i*14 + j]);
    }
    for (int i = 0; i < sp_num; i++) {
        MPI_PRINTF("\n Specie[%d]'s T_range are: \n", i);
        for (int j = 0; j < 3; j++)
            MPI_PRINTF("%e\t", species_const_c.T_range[i*3 + j]);
    }
#endif //DEBUG

}

/** Destory class species_d.
 */
species_d::~species_d() {
    CUDACHECK(hipFree(this->species_const_d.sp_W));
    CUDACHECK(hipFree(this->species_const_d.sp_nasa));
    CUDACHECK(hipFree(this->species_const_d.T_range));
}