/*******************************************************************************************
 * This file contains the parameters and function implementation related to ODE solver
 ******************************************************************************************/
#include <math.h>

#include "opencc.h"

#include "species.h"
#include "thermoFluid.h"
#include "reactions.h"
#include "jacobian.h"
#include "seulex.h"
#include "thermoFluid_kernel.h"

/** Create related scopes.
 *  \param[in] elements_const_d_ref Reference value used to assign value to parameter elements_d_ptr,
 *                                  see elements_const for more.
 *  \param[in] species_const_d_ref  Reference value used to assign value to parameter species_d_ptr,
 *                                  see species_const for more.
 *  \param[in] thermo_ptr_d         Pointer to device memory used for thermophysical 
 *                                  properties calculation, memory allocation and settings in set_thermoFluid.
 *
 *  \param[in] reactions_ptr_d_ref  Pointer to device memory.
 */
void opencc_create(
    elements_const *elements_const_d_ref,
    species_const *species_const_d_ref,
    thermo_ptr *thermo_ptr_d,
    reactions_ptr *reactions_ptr_d_ref
){
// Create related scopes.

    for(int i = 0; i < Stream_num; i++) {

        hipStreamCreate(&Stream_opencc[i]);

        hipEventCreate(&Event[i]);
    }

    create_elements(elements_const_d_ref);

    create_species(species_const_d_ref);

    create_thermoFluid(thermo_ptr_d);

    create_reactions(reactions_ptr_d_ref);

    create_seulex();

    hipDeviceSynchronize();
}

/** Solution of chemical ODE.
 *  \param[in] t_init the intial time
 *  \param[in] t_end the end time
 */
void opencc_ode(
    REAL t_init,
    REAL t_end
){
    size_t size   = thermoFluid_d_ptr->thermo_ptr_d.size;
    size_t sp_num = species_d_ptr->species_const_d.sp_num;

    t_end_h = t_end;

    seulex_d_ptr->dt_c = t_init;
    
    //cuda_mem_value_init(seulex_d_ptr->dt_c, seulex_d_ptr->dt_c_d, 1, 1, size, block_set_J0);
    cuda_copy(seulex_d_ptr->dt_c_new_d, seulex_d_ptr->dt_c_d, 1, 1, size, block_set_J0);
    cuda_mem_value_init(0.0, dt_sum_d, 1, 1, size, block_set_J0);

    thermo_ptr thermo_ptr_h;

    int steps = 0;

    REAL flag = 1.;

    REAL *T_min = (REAL *)malloc(sizeof(REAL));
    REAL *T_min_d = nullptr; CUDACHECK(hipMalloc((void**)&T_min_d, sizeof(REAL)));

    REAL *T_max = (REAL *)malloc(sizeof(REAL));
    REAL *T_max_d = nullptr; CUDACHECK(hipMalloc((void**)&T_max_d, sizeof(REAL)));

	my_timer_opencc tm, tm_step;

    //-----------------------  Update X from Y 
    update_fraction(Y_TO_X);

    //-----------------------  Update thermo from T, P, Y 
    set_state_TPY();

    update_c_from_T_P_h(thermoFluid_d_ptr->thermo_ptr_d.c, thermoFluid_d_ptr->thermo_ptr_d.T,
                thermoFluid_d_ptr->thermo_ptr_d.P);
                
    //-----------------------  Calculation the chemical reaction rate 
    //cal_react_rate();

    //-----------------------  Compute Jacobian form Y, T 
    Jacobian();

	tm.start();

    do {
	    tm_step.start();

    //-----------------------  ODE solver
        seulex_solver(t_end, &flag);
			
        c_constraint_h(thermoFluid_d_ptr->thermo_ptr_d.c);

    //----------------------- Update T from ha.

        update_T_P_from_c_h(thermoFluid_d_ptr->thermo_ptr_d.c, thermoFluid_d_ptr->thermo_ptr_d.T,
                thermoFluid_d_ptr->thermo_ptr_d.P);

	    tm_step.stop();

        steps += 1;

        MPI_PRINTF("\nSeulex's steps is %d, t_sum_min is %e, Timecost of GPU (step) = %lf s\n", 
            steps, *(seulex_d_ptr->dt_sum_min), tm_step.time_use);
#ifdef PERF
    exit(0);
#endif //PERF
    } while (flag == 1.);

    cal_react_rate();

    h_constraint();

    update_fraction(X_TO_Y);

	tm.stop();

    MPI_PRINTF("\nTotal Timecost of GPU (ODE) = %lf s\n", tm.time_use);


#ifdef DEBUG    
    //get_thermoFluid(&thermo_ptr_h, THERMO_ALL);

    //for (int i = 0; i < size; i++) {
    //    for (int j = 0; j < sp_num + 2; j++) {
    //        MPI_PRINTF("c[%d][%d] is %e\t", i, j, *(thermo_ptr_h.c + j + i*(sp_num + 2)));
    //    }
    //    MPI_PRINTF("\n");
    //}
#endif //DEBUG*/
}


void opencc_ode_all(REAL *T, REAL *P, REAL *Y, REAL t_init, REAL t_end, DATA_MODE op) {
    size_t size   = thermoFluid_d_ptr->size_origin;
    size_t sp_num = species_d_ptr->species_const_d.sp_num;

    thermoFluid_d_ptr->set_TPY(T, P, Y, sp_num, size, op);

    opencc_ode(t_init, t_end);

    thermoFluid_d_ptr->get_Y(Y, sp_num, size, op);

}
