#include "species.h"
#include "thermoFluid.h"
#include "reactions.h"
#include "jacobian.h"
#include "seulex.h"

/** Create elements scope and assign parameters with reference values.
 *
 *  \param[in] elements_const_d_ref  Reference value used to assign value to parameter elements_d_ptr,
 *                                   see elements_const for more.
 */
void create_elements(elements_const *elements_const_d_ref) {
    elements_d_ptr = new elements_d(elements_const_d_ref);
}

/** Detele elements scope.
 */
void delete_elements() {
    delete elements_d_ptr;
}

/** Create species scope and assign parameters with reference values 
 *
 *  \param[in] species_const_d_ref  Reference value used to assign value to parameter species_d_ptr,
 *                                  see species_const for more.
 */
void create_species(species_const *species_const_d_ref) {
    species_d_ptr = new species_d(species_const_d_ref);
}

/** Detele species scope.
 */
void delete_species() {
    delete species_d_ptr;
}

/** Create the scope for thermophysical property-related calculations.
 *
 *  \param[in] thermo_ptr_d  Pointer to device memory used for thermophysical 
 *                           properties calculation, memory allocation and settings in set_thermoFluid.
 */
void create_thermoFluid(thermo_ptr *thermo_ptr_d) {
    thermoFluid_d_ptr = new thermoFluid_d(thermo_ptr_d);
}

/** Destory the scope for thermophysical property-related calculations.
 */
void destory_thermoFluid() {
    delete thermoFluid_d_ptr; 
}

/** set the thermo_ptr_d from thermo_ptr_h.
 *  
 *  \param[in] sp_num        The number of species. (host)
 *  \param[in] size          Number of grid points. (host)
 *  \param[in] thermo_ptr_h  The memory address of the host side corresponding to 
 *                           the packaged storage thermo parameters. (host)
 *  \param[in] op            Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void set_thermoFluid(int sp_num, int size, thermo_ptr *thermo_ptr_h, THERMO_SET_MODE op){
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_d_set(sp_num, size, thermo_ptr_h, op);
    }
}

/** Query the thermo_ptr_d.
 *  
 *  \param[out] thermo_ptr_h  The memory address of the host side corresponding to 
 *                            the packaged storage thermo parameters. (host)
 *  \param[in]  op            Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void get_thermoFluid(thermo_ptr *thermo_ptr_h, THERMO_SET_MODE op){
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_d_get(thermo_ptr_h, op);
    }
}

/** Update X from Y or update Y from X.
 *  \param[in] op            Data processing mode, see UPDATE_FRAC_MODE for more. (host) 
 */
void update_fraction(UPDATE_FRAC_MODE op){
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_update_fraction(op);
    }
}

/** Equifunctional constraint.
 */
void h_constraint() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_h_constraint();
    }
}

/** c constraint.
 */
void c_constraint() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_c_constraint();
    }
}

/** Update thermo (rho, cp, psi, c, ha) from T, P, Y.
 */
void set_state_TPY() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_set_state_TPY();
    }
}

/** Copy T, P, Y to device.
 */
void set_TPY(REAL *T, REAL *P, REAL *Y, int sp_num, int size, DATA_MODE op) {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->set_TPY(T, P, Y, sp_num, size, op);
    }
}

/** get Y from device.
 */
void get_Y(REAL *Y, int sp_num, int size, DATA_MODE op) {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->get_Y(Y, sp_num, size, op);
    }
}

/** Compute hc.
 */
void compute_hc_mass() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_compute_hc_mass();
    }
}

/** Update thermo (rho, cp, psi, c, ha) from T, P, Y.
 */
void set_state_hPY() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_set_state_hPY();
    }
}

/** Update c from T, P.
 */
void update_c_from_TP() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_update_c_from_TP();
    }
}

/** Update T, P from c.
 */
void update_TP_from_c() {
    if (thermoFluid_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_thermoFluid first.\033[0m\n");
    } else {
        thermoFluid_d_ptr->thermoFluid_update_TP_from_c();
    }
}

/** Construct the scope for reactions_d.
 *  
 *  \param[in] reactions_ptr_d_ref  Pointer to device memory.
 */
void create_reactions(reactions_ptr *reactions_ptr_d_ref) {
    reactions_d_ptr = new reactions_d(reactions_ptr_d_ref);
}

/** Destory the scope for reactions_d.
 */
void delete_reactions() {
    delete reactions_d_ptr;
}

/** Set class reactions_d.
 *  
 *  \param[in] react_num        chemical reaction number. (host)
 *  \param[in] reactions_ptr_h  The memory address of the host side corresponding to 
 *                              the packaged storage reactions parameters. (host)
 *  \param[in] op               Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void set_reactions(int react_num, reactions_ptr *reactions_ptr_h, REACTION_SET_MODE op) {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->reactions_d_set(react_num, reactions_ptr_h, op);
    }
}

/** Query class reactions_d.
 *  
 *  \param[in] reactions_ptr_h  The memory address of the host side corresponding to 
 *                              the packaged storage reactions parameters. (host)
 *  \param[in] op               Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void get_reactions(reactions_ptr *reactions_ptr_h, REACTION_SET_MODE op) {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->reactions_d_get(reactions_ptr_h, op);
    }
}

/** Calculation of chemical reaction rates.
 */
void cal_react_rate() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->cal_react_rate(thermoFluid_d_ptr->thermo_ptr_d.c, reactions_d_ptr->reactions_ptr_d.dcdt);
    }
}

/** Copy data of J form device to host.
 */
void get_jacobian() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->jacobian_d_get();
    }
}

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ0() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->updateJ0();
    }
}

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ1() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->updateJ1(thermoFluid_d_ptr->thermo_ptr_d.c);
    }
}

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ2() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->updateJ2(thermoFluid_d_ptr->thermo_ptr_d.c);
    }
}

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ3() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->updateJ3(thermoFluid_d_ptr->thermo_ptr_d.c);
    }
}

/** Compute Jacobian.
 */
void Jacobian() {
    if (reactions_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_reactions first.\033[0m\n");
    } else {
        reactions_d_ptr->jacobian(thermoFluid_d_ptr->thermo_ptr_d.c);
    }
}

/** Construct the scope for seleux_d.
 */
void create_seulex() {
    seulex_d_ptr = new seulex_d();
}

/** Set data of J form host to device.
 *  \param[in] J_ref   host ptr used to set J.
 */
void set_seulex_J(REAL *J_ref) {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_set_J(J_ref);
    }
}

/** Set data of dy form host to device.
 *  \param[in] dy_ref   host ptr used to set dy.
 */
void set_seulex_dy(REAL *dy_ref) {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_set_dy(dy_ref);
    }
}

/** Copy data of J form device to host.
 */
void get_seulex_J() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_get_J();
    }
}

/** Copy data of y_seq form device to host.
 */
void get_seulex_y_seq(REAL *y_seq_h) {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->get_y_seq(y_seq_h);
    }
}

/** Compute A.
 */
void compute_seulex_A() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->compute_A();
    }
}

/** PLU decomposition.
 */
void seulex_PLU_decomposition() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->PLU_decomposition();
    }
}

/** Slove linear system.
 */
void seulex_solve_linear_system() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->solve_linear_system();
    }
}

/** get PLU decomposition data.
 */
void get_seulex_PLU() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_get_PLU();
    }
}

/** Copy data of A form device to host.
 */
void get_seulex_A() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_get_A();
    }
}

/** Copy data of dy form device to host.
 */
void get_seulex_dy() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_d_get_dy();
    }
}

/** compute_scale of seulex.
 */
void seulex_compute_scale() {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->compute_scale();
    }
}

/** seulex.
 */
void seulex(int k) {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seul(k);
    }
}

/** seulex_solver.
 */
void seulex_solver(REAL t_end, REAL *flag) {
    if (seulex_d_ptr == nullptr) {
        MPI_PRINTF("\033[31mWORRY!!! Please create_seulex first.\033[0m\n");
    } else {
        seulex_d_ptr->seulex_solver(t_end, flag);
    }
}

