/*******************************************************************************************
 * This file contains the parameters and function implementation related to jacobian
 ******************************************************************************************/

#include "thermoFluid.h"
#include "species.h"

#include "jacobian.h"
#include "reactions.h"
#include "jacobian_kernel.h"

jacobian_d::jacobian_d() {

}

jacobian_d::~jacobian_d() {

    CUDACHECK(hipFree(J_d));

    CUDACHECK(hipFree(tmp_ptr));
}

void jacobian_d::jacobian_init() {
    size_t size      = thermoFluid_d_ptr->thermo_ptr_d.size;
    size_t sp_num    = species_d_ptr->species_const_d.sp_num;
    size_t react_num = reactions_ptr_d.react_num;

    size_t pitch;

    J_h = (REAL*)malloc((sp_num+2)*(sp_num+2)*size*sizeof(REAL));

    CUDACHECK(hipMalloc((void**)&J_d, (sp_num+2)*(sp_num+2)*sizeof(REAL)*size));

    cuda_mem_value_init(0.0, J_d, (sp_num+2)*(sp_num+2), (sp_num+2)*(sp_num+2), size, block_set_J0);

    pitch = (sp_num+2)*(sp_num+2);
    CUDACHECK(hipMemcpyToSymbol(HIP_SYMBOL(HIP_SYMBOL(pitch_J_d)), &pitch, sizeof(size_t), 0, hipMemcpyHostToDevice));

    pitch_J = pitch;

    CUDACHECK(hipMalloc((void**)&tmp_ptr, sp_num*sizeof(REAL)*size));

    cuda_mem_value_init(0.0, tmp_ptr, sp_num, sp_num, size, block_set_J0);
}

/** Copy data of J form device to host.
 */
void jacobian_d::jacobian_d_get() {
    size_t size      = thermoFluid_d_ptr->thermo_ptr_d.size;
    size_t sp_num    = species_d_ptr->species_const_d.sp_num;

    DeviceDataget(J_d, J_h,  size*(sp_num+2)*(sp_num+2)*sizeof(REAL));

#ifdef DEBUG    
    MPI_PRINTF("\n");
    int index = 0;
    for(int s = 0; s < size; s++){
        for(int i = 0; i < sp_num+2; i++){
            for(int j = 0; j < sp_num+2; j++){
                index = j+(sp_num+2)*i+s*(sp_num+2)*(sp_num+2);
                MPI_PRINTF("%e  ", *(J_h+index));
            }
            MPI_PRINTF("\n");
        }
    }
#endif //DEBUG
}

/** Update Jacobian form c, kf, kr.
 */
void jacobian_d::updateJ0() {

    size_t size      = thermoFluid_d_ptr->thermo_ptr_d.size;
    size_t sp_num    = species_d_ptr->species_const_d.sp_num;
    size_t react_num = reactions_ptr_d.react_num;

    REAL *tmp_ptr0 = tmp_ptr;

    cuda_mem_value_init(0.0, J_d, (sp_num+2)*(sp_num+2), (sp_num+2)*(sp_num+2), size, block_set_J0);

    computeJ0(reactions_ptr_d.kf, reactions_ptr_d.kr, reactions_ptr_d.vf, reactions_ptr_d.vr, reactions_ptr_d.react_c,
    thermoFluid_d_ptr->thermo_ptr_d.c, J_d);
}

/** Update Jacobian form c, kf, kr.
 */
void jacobian_d::updateJ1(REAL *y0) {

    size_t size      = thermoFluid_d_ptr->thermo_ptr_d.size;
    size_t sp_num    = species_d_ptr->species_const_d.sp_num;
    size_t react_num = reactions_ptr_d.react_num;

    computeJ1(reactions_ptr_d.react_type, reactions_ptr_d.fall_type, reactions_ptr_d.q,
              thermoFluid_d_ptr->thermo_ptr_d.T, reactions_ptr_d.tb_coeffs, 
              reactions_ptr_d.fall_coeffs, y0,
              J_d, reactions_ptr_d.v_net, reactions_ptr_d.kf_low,
              reactions_ptr_d.kf);
}

/** Update Jacobian form c, kf, kr.
 */
void jacobian_d::updateJ2(REAL *y0) {

    compute_dBdT_h(thermoFluid_d_ptr->thermo_ptr_d.T,
                   species_d_ptr->species_const_d.T_range,
                   species_d_ptr->species_const_d.sp_nasa,
                   tmp_ptr);

    compute_J2_h(tmp_ptr, 
    reactions_ptr_d.q,
    reactions_ptr_d.v_net,
    reactions_ptr_d.kf_low,
    reactions_ptr_d.kf,
    reactions_ptr_d.kr,
    thermoFluid_d_ptr->thermo_ptr_d.T,
    reactions_ptr_d.abe,
    reactions_ptr_d.react_type,
    reactions_ptr_d.tb_coeffs,
    y0,
    reactions_ptr_d.fall_type,
    reactions_ptr_d.fall_coeffs,
    reactions_ptr_d.vr,
    reactions_ptr_d.vf,
    J_d);     
        
}

/** Update Jacobian form c, kf, kr.
 */
void jacobian_d::updateJ3(REAL *y0) {

    thermoFluid_d_ptr->thermoFluid_compute_ha_cp();

    compute_J3_h(thermoFluid_d_ptr->thermo_ptr_d.T,
    species_d_ptr->species_const_d.T_range,
    species_d_ptr->species_const_d.sp_nasa,
    y0,
    thermoFluid_d_ptr->thermo_ptr_d.sp_cp_mole,
    thermoFluid_d_ptr->thermo_ptr_d.sp_ha_mole,
    reactions_ptr_d.dcdt,
    J_d);
}

/** Compute Jacobian.
 */
void jacobian_d::jacobian(REAL *y0) {

#ifdef PERF
	my_timer_opencc tm;
    hipDeviceSynchronize();
	tm.start();
#endif //PERF

    reactions_d_ptr->cal_react_rate(y0, reactions_ptr_d.dcdt);

#ifdef PERF
    hipDeviceSynchronize();
	tm.stop();
    MPI_PRINTF("\ncal_react_rate Timecost of GPU (ODE) = %lf s (JACOBIAN)\n", tm.time_use);
	tm.start();
#endif //PERF

    updateJ0();

#ifdef PERF
    hipDeviceSynchronize();
	tm.stop();
    MPI_PRINTF("\nupdateJ0 Timecost of GPU (ODE) = %lf s\n", tm.time_use);
	tm.start();
#endif //PERF

    updateJ1(y0);

#ifdef PERF
    hipDeviceSynchronize();
	tm.stop();
    MPI_PRINTF("\nupdateJ1 Timecost of GPU (ODE) = %lf s\n", tm.time_use);
	tm.start();
#endif //PERF

    updateJ2(y0);

#ifdef PERF
    hipDeviceSynchronize();
	tm.stop();
    MPI_PRINTF("\nupdateJ2 Timecost of GPU (ODE) = %lf s\n", tm.time_use);
	tm.start();
#endif //PERF

    updateJ3(y0);

#ifdef PERF
    hipDeviceSynchronize();
	tm.stop();
    MPI_PRINTF("\nupdateJ3 Timecost of GPU (ODE) = %lf s\n", tm.time_use);
#endif //PERF
}


