#ifndef _OPENCC_H
#define _OPENCC_H

#include <stdio.h>  
#include <stdlib.h>  

#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

/** Set calculation precision globally.
 */
#define REAL double

/** A structure for storing the name and molar mass of an element.
 */
typedef struct configItem_
{
    char ele_names[1000];   //The name of the element.
    REAL ele_W;             //The molar mass of an element.
} configItem;

/** A structure for storing all elements information.
 */
typedef struct elements_const_
{
    configItem *elements = nullptr; //A structure that holds the name and molar mass of an element, ([species_num])

    int ele_num = 0;                 //elements_num, (scalar)
} elements_const;

/** Create elements scope and assign parameters with reference values.
 *
 *  \param[in] elements_const_d_ref  Reference value used to assign value to parameter elements_d_ptr,
 *                                   see elements_const for more.
 */
void create_elements(elements_const *elements_const_d_ref);

/** Detele elements scope.
 */
void delete_elements();

/** A structure for storing all species information.
 *  \details The physical field (temperature, mole fraction, etc.) is not required as the input 
 *  update quantity.
 */
typedef struct species_const_
{
    REAL *sp_W = nullptr;           //kg/kmol, ([species_num])

    char *sp_name = nullptr;        //species's name, ([species_num][100 byte])

    /* the first seven are low temperature region, the last seven are high temperature region.
     */
    REAL *sp_nasa = nullptr;        //Nasa coefficient, ([species_num][14])

    REAL *T_range = nullptr;        //Temperature range, ([species_num][3])

    int sp_num = 0;                 //species_num, (scalar)
} species_const;

/** Create species scope and assign parameters with reference values 
 *
 *  \param[in] species_const_d_ref  Reference value used to assign value to parameter species_d_ptr,
 *                                  see species_const for more.
 */
void create_species(species_const *species_const_d_ref);

/** Detele species scope.
 */
void delete_species();

/** A structure for storing thermophysical parameters.
 */
typedef struct thermo_ptr_
{ 
//-------------------------- thermal physical parameters
    REAL *T = nullptr;          //Temperature, K, (scalar)
    REAL *P = nullptr;          //Pressure, Pa, (scalar) 
    REAL *Y = nullptr;          //mass fraction, ([species_num])
    REAL *X = nullptr;          //mole fraction, ([species_num])
    //---------------------- Above data from CFD

    REAL *rho     = nullptr;    //Density, kg/m^3, (scalar) 
    REAL *W_mix   = nullptr;    //(scalar)
    REAL *ha_mass = nullptr;    //total enthalpy per unit density, J/(kg/m^3), (scalar)
    REAL *ha_mole = nullptr;    //total enthalpy per unit concentration, J/(kmol/m^3), (scalar)
    REAL *hc_mass = nullptr;    //chemical enthalpy per unit density, (scalar)
    REAL *hc_mole = nullptr;    //chemical enthalpy per unit mole, (scalar)
    REAL *hs_mass = nullptr;    //sensible enthalpy per unit density, (scalar)
    REAL *hs_mole = nullptr;    //sensible enthalpy per unit mole, (scalar)
    REAL *cp_mole = nullptr;    //specific heat at constant pressure, (scalar)
    REAL *psi     = nullptr;    //compression coefficient, (scalar) 
    
    REAL *c          = nullptr; //concentrations + T + P, kmol/m^3, ([species_num+2])
    REAL *sp_ha_mole = nullptr; //Mole ratio total enthalpy of each component, ([species_num])
    REAL *sp_cp_mole = nullptr; //([species_num])

//-------------------------- constant parameters
    int sp_num = 0;    //species_num, (scalar, const)
    int size   = 0;    //grids_num, (scalar, const)
} thermo_ptr;

/** An enumeration used with the calculation of thermophysical properties.
 */
typedef enum 
{
    THERMO_ALL      = 0,
    T_MODE          = 1,
    P_MODE          = 2,
    Y_MODE          = 3,
    X_MODE          = 4,
    rho_MODE        = 5,
    W_mix_MODE      = 6,
    ha_mass_MODE    = 7,
    ha_mole_MODE    = 8,
    hc_mass_MODE    = 9,
    hc_mole_MODE    = 10,
    hs_mass_MODE    = 11,
    hs_mole_MODE    = 12,
    cp_mole_MODE    = 13,
    psi_MODE        = 14,
    c_MODE          = 15,
    sp_ha_mole_MODE = 16,
    sp_cp_mole_MODE = 17,
    sp_num_MODE     = 18
} THERMO_SET_MODE;

/** Create the scope for thermophysical property-related calculations.
 *
 *  \param[in] thermo_ptr_d  Pointer to device memory used for thermophysical 
 *                           properties calculation, memory allocation and settings in set_thermoFluid.
 */
void create_thermoFluid(thermo_ptr *thermo_ptr_d);

/** Destory the scope for thermophysical property-related calculations.
 */
void destory_thermoFluid();

/** set the thermo_ptr_d from thermo_ptr_h.
 *  
 *  \param[in] sp_num        The number of species. (host)
 *  \param[in] size          Number of grid points. (host)
 *  \param[in] thermo_ptr_h  The memory address of the host side corresponding to 
 *                           the packaged storage thermo parameters. (host)
 *  \param[in] op            Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void set_thermoFluid(int sp_num, int size, thermo_ptr *thermo_ptr_h, THERMO_SET_MODE op);

/** Query the thermo_ptr_d.
 *  \param[out] thermo_ptr_h  The memory address of the host side corresponding to 
 *                            the packaged storage thermo parameters. (host)
 *  \param[in] op             Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void get_thermoFluid(thermo_ptr *thermo_ptr_h, THERMO_SET_MODE op);

/** An enumeration used with the calculation of thermophysical properties.
 */
typedef enum 
{
    Y_TO_X = 0, //update X from Y
    X_TO_Y = 1  //Update Y from X
} UPDATE_FRAC_MODE;

typedef enum 
{
    CPU = 0, //copy data from CPU
    GPU = 1  //copy data from GPU
} DATA_MODE;

/** Copy T, P, Y to device.
 */
void set_TPY(REAL *T, REAL *P, REAL *Y, int sp_num, int size, DATA_MODE op);

/** get Y from device.
 */
void get_Y(REAL *Y, int sp_num, int size, DATA_MODE op);

/** Update X from Y or update Y from X.
 *  \param[in] op            Data processing mode, see UPDATE_FRAC_MODE for more. (host) 
 */
void update_fraction(UPDATE_FRAC_MODE op);

/** Equifunctional constraint.
 */
void h_constraint();

/** c constraint.
 */
void c_constraint();

/** Compute hc.
 */
void compute_hc_mass();

/** Update thermo (rho, cp, psi, c, ha) from T, P, Y.
 */
void set_state_TPY();

/** Update T from ha.
 */
void set_state_hPY();

/** Update c from T, P.
 */
void update_c_from_TP();

/** Update T, P from c.
 */
void update_TP_from_c();

/** A structure for storing all species information.
 *  \details The physical field (temperature, mole fraction, etc.) is not required as the input 
 *  update quantity.
 */
typedef struct reactions_ptr_
{
//-------------------------- initial const parameters
    REAL *vf        = nullptr;          //forward chemical reaction equivalent number, ([sp_num][react_num])
    REAL *vr        = nullptr;          //reverse chemical reaction equivalent number, ([sp_num][react_num])
    REAL *v_net     = nullptr;          //net chemical reaction equivalent number, ([sp_num][react_num])
    /* Three-body coefficients (used for three-body reactions and decay reactions), 
     * ([sp_num][react_num])
     */
    REAL *tb_coeffs = nullptr; 

    REAL *order     = nullptr;          //reaction order, ([react_num])

    REAL *react_type = nullptr;         //reaction types: 0 - basic; 1 - 3 body; 2 - decay, ([react_num])
    REAL *is_rev     = nullptr;         //Is it reversible? : 1 - yes; 0 - no; ([react_num])
    REAL *fall_type  = nullptr;         //decay response type: 1 - Lindemann; 2 - Troe; 3 - SRI, ([react_num])

    /* ABE three-term coefficient, basic reaction and three-body reaction use the first three, 
     * attenuation reaction use all, ([react_num][6])
     */
    REAL *abe = nullptr;

    /* Attenuation coefficient (Troe uses,a,T1,T2,T3 four parameters, SRI uses 5, Lindermann does not 
     * use, here according to the maximum SRI design), ([react_num][5])
     */
    REAL *fall_coeffs = nullptr;

//-------------------------- const parameters
    REAL *kf = nullptr;                 //chemical equilibrium constant of forward reaction, ([gird_num][react_num])
    REAL *kf_low = nullptr; 
    REAL *kr = nullptr;                 //chemical equilibrium constant of reverse reaction, ([gird_num][react_num])

//-----------------------------------------------------------
    REAL *Rf = nullptr;                 //reaction rate of basal forward reaction, kmol/m3/s, ([gird_num][react_num])
    REAL *Rr = nullptr;                 //reaction rate of basal reverse reaction, kmol/m3/s, ([gird_num][react_num])
    REAL *R_net = nullptr;              //reaction rate of basal net reaction, kmol/m3/s, ([gird_num][react_num]), R_net = Rf - Rr
    REAL *react_c = nullptr;            //response modification factor, ([gird_num][react_num]) 
    
    REAL *q = nullptr;                  //Net reaction rate of reaction, kmol/m3/s, ([gird_num][react_num]) , q = react_c + R_net
    REAL *sp_net_rate = nullptr;        //Net reaction rate of species, kmol/m3/s, ([gird_num][sp_num]) 

    REAL *dcdt = nullptr;               //([gird_num][sp_num + 2])
    
    int react_num   = 0;                //chemical reaction number
} reactions_ptr;

/** An enumeration used with the calculation of reactions.
 */
typedef enum 
{
    REACTIONS_ALL    = 0,
    vf_MODE          = 1,
    vr_MODE          = 2,
    v_net_MODE       = 3,
    react_type_MODE  = 4,
    is_rev_MODE      = 5,
    abe_MODE         = 6,
    tb_coeffs_MODE   = 7,
    fall_type_MODE   = 8,
    fall_coeffs_MODE = 9,
    order_MODE       = 10,
    kf_MODE          = 11,
    kf_low_MODE      = 12,
    kr_MODE          = 13,
    Rf_MODE          = 14,
    Rr_MODE          = 15,
    R_net_MODE       = 16,
    react_c_MODE     = 17,
    q_MODE           = 18,
    sp_net_rate_MODE = 19,
    dcdt_MODE        = 20
} REACTION_SET_MODE;

/** Construct the scope for reactions_d.
 *  
 *  \param[in] reactions_ptr_d_ref  Pointer to device memory.
 */
void create_reactions(reactions_ptr *reactions_ptr_d_ref);

/** Destory the scope for reactions_d.
 */
void delete_reactions();

/** Set class reactions_d.
 *  
 *  \param[in] react_num        chemical reaction number. (host)
 *  \param[in] reactions_ptr_h  The memory address of the host side corresponding to 
 *                              the packaged storage reactions parameters. (host)
 *  \param[in] op               Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void set_reactions(int react_num, reactions_ptr *reactions_ptr_h, REACTION_SET_MODE op);

/** Query class reactions_d.
 *  
 *  \param[in] reactions_ptr_h  The memory address of the host side corresponding to 
 *                              the packaged storage reactions parameters. (host)
 *  \param[in] op               Data processing mode, see THERMO_SET_MODE for more. (host) 
 */
void get_reactions(reactions_ptr *reactions_ptr_h, REACTION_SET_MODE op);

/** Calculation of chemical reaction rates.
 */
void cal_react_rate();

/** Copy data of J form device to host.
 */
void get_jacobian();

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ0();

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ1();

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ2();

/** Update Jacobian form c, kf, kr.
 *  
 *  \details C(chemical reaction rate) should be set throught set_thermoFluid before launch this kernel.
 */
void updateJ3();

/** Compute Jacobian.
 */
void Jacobian();

/** Construct the scope for seleux_d.
 */
void create_seulex();

/** Set data of J form host to device.
 *  \param[in] J_ref   host ptr used to set J.
 */
void set_seulex_J(REAL *J_ref);

/** Set data of dy form host to device.
 *  \param[in] dy_ref   host ptr used to set dy.
 */
void set_seulex_dy(REAL *dy_ref);

/** Copy data of J form device to host.
 */
void get_seulex_J();

/** Copy data of y_seq form device to host.
 */
void get_seulex_y_seq(REAL *y_seq_h);

/** Compute A.
 */
void compute_seulex_A();

/** PLU decomposition.
 */
void seulex_PLU_decomposition();

/** Slove linear system.
 */
void seulex_solve_linear_system();

/** get PLU decomposition data.
 */
void get_seulex_PLU();

/** Copy data of A form device to host.
 */
void get_seulex_A();

/** Copy data of dy form device to host.
 */
void get_seulex_dy();

/** compute_scale of seulex
 */
void seulex_compute_scale();

/** seulex.
 */
void seulex(int k);

/** seulex_solver.
 */
void seulex_solver(REAL t_end, REAL *flag);

/** Create related scopes.
 *  \param[in] elements_const_d_ref Reference value used to assign value to parameter elements_d_ptr,
 *                                  see elements_const for more.
 *  \param[in] species_const_d_ref  Reference value used to assign value to parameter species_d_ptr,
 *                                  see species_const for more.
 *  \param[in] thermo_ptr_d         Pointer to device memory used for thermophysical 
 *                                  properties calculation, memory allocation and settings in set_thermoFluid.
 *
 *  \param[in] reactions_ptr_d_ref  Pointer to device memory.
 */
void opencc_create(
    elements_const *elements_const_d_ref,
    species_const *species_const_d_ref,
    thermo_ptr *thermo_ptr_d,
    reactions_ptr *reactions_ptr_d_ref
);

/** Solution of chemical ODE.
 *  \param[in] t_init the intial time
 *  \param[in] t_end the end time
 */
void opencc_ode(
    REAL t_init,
    REAL t_end
);

/** Init opencc.
 *  \param[in]      yaml_name  The position of mechanism file.
 *  \param[in]      size       the number of grids.
 */
void opencc_ode_init(char *yaml_name, int size, REAL *T, REAL *P, REAL *Y);

/** Solution of chemical ODE.
 *  \param[in]      T      the pointer about temperature. (k).
 *  \param[in]      P      the pointer about pressure. (pa).
 *  \param[in, out] Y      the pointer about mass fraction. [size][sp_num].
 *  \param[in]      t_init the intial time.
 *  \param[in]      t_end  the end time.
 *  \param[in]      op     enumeration of data sources, 'CPU' or 'GPU'.
 */
void opencc_ode_all(REAL *T, REAL *P, REAL *Y, REAL t_init, REAL t_end, DATA_MODE op);


#if defined(__cplusplus)
}
#endif //__cplusplus

#endif //_OPENCC_H
