/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Description
    Cell to face interpolation scheme. Included in fvMesh.

\*---------------------------------------------------------------------------*/

#include "gpusurfaceInterpolation.H"
#include "gpufvMesh.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"
#include "coupledgpuFvPatch.H"
#include "basicgpuFvGeometryScheme.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(gpusurfaceInterpolation, 0);
}


// * * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * //

void Foam::gpusurfaceInterpolation::clearOut()
{
    weights_.clear();
    deltaCoeffs_.clear();
    nonOrthDeltaCoeffs_.clear();
    nonOrthCorrectionVectors_.clear();
}


// * * * * * * * * * * * * * * * * Constructors * * * * * * * * * * * * * * //

Foam::gpusurfaceInterpolation::gpusurfaceInterpolation(const gpufvMesh& fvm)
:
    mesh_(fvm),
    weights_(nullptr),
    deltaCoeffs_(nullptr),
    nonOrthDeltaCoeffs_(nullptr),
    nonOrthCorrectionVectors_(nullptr)
{}


// * * * * * * * * * * * * * * * * Destructor * * * * * * * * * * * * * * * //

Foam::gpusurfaceInterpolation::~gpusurfaceInterpolation()
{
    clearOut();
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

const Foam::gpufvGeometryScheme& Foam::gpusurfaceInterpolation::geometry() const
{
    if (!geometryPtr_.valid())
    {
        geometryPtr_ = gpufvGeometryScheme::New
        (
            mesh_,
            mesh_.hostmesh().schemesDict().subOrEmptyDict("geometry"),
            basicgpuFvGeometryScheme::typeName
        );
    }

    return geometryPtr_();
}


void Foam::gpusurfaceInterpolation::geometry(tmp<gpufvGeometryScheme>& schemePtr)
{
    geometryPtr_ = schemePtr;
}


const Foam::surfaceScalargpuField& Foam::gpusurfaceInterpolation::weights() const
{
    if (!weights_.valid())
    {
        weights_.set(geometry().weights().ptr());
    }

    return weights_();
}


const Foam::surfaceScalargpuField& Foam::gpusurfaceInterpolation::deltaCoeffs() const
{
    if (!deltaCoeffs_.valid())
    {
        deltaCoeffs_.set(geometry().deltaCoeffs().ptr());
    }

    return deltaCoeffs_();
}


const Foam::surfaceScalargpuField&
Foam::gpusurfaceInterpolation::nonOrthDeltaCoeffs() const
{
    if (!nonOrthDeltaCoeffs_.valid())
    {
        nonOrthDeltaCoeffs_.set(geometry().nonOrthDeltaCoeffs().ptr());
    }

    return nonOrthDeltaCoeffs_();
}


const Foam::surfaceVectorgpuField&
Foam::gpusurfaceInterpolation::nonOrthCorrectionVectors() const
{
    if (!nonOrthCorrectionVectors_.valid())
    {
        nonOrthCorrectionVectors_.set
        (
            geometry().nonOrthCorrectionVectors().ptr()
        );
    }

    return nonOrthCorrectionVectors_();
}


bool Foam::gpusurfaceInterpolation::movePoints()
{
    if (debug)
    {
        Pout<< "surfaceInterpolation::movePoints() : "
            << "Updating geometric properties using the fvGeometryScheme"
            << endl;
    }

    // Do any primitive geometry calculation
    const_cast<gpufvGeometryScheme&>(geometry()).movePoints();

    weights_.clear();
    deltaCoeffs_.clear();
    nonOrthDeltaCoeffs_.clear();
    nonOrthCorrectionVectors_.clear();

    return true;
}


void Foam::gpusurfaceInterpolation::updateGeom()
{
    if (debug)
    {
        Pout<< "surfaceInterpolation::updateGeom() : "
            << "Updating geometric properties" << endl;
    }

    const_cast<gpufvGeometryScheme&>(geometry()).movePoints();
}


// ************************************************************************* //
