/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::linear

Group
    grpFvSurfaceInterpolationSchemes

Description
    Central-differencing interpolation scheme class

SourceFiles
    linear.C

\*---------------------------------------------------------------------------*/

#ifndef gpulinear_H
#define gpulinear_H

#include "gpusurfaceInterpolationScheme.H"
#include "volgpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class linear Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulinear
:
    public gpusurfaceInterpolationScheme<Type>
{
    // Private Member Functions

        //- No copy assignment
        void operator=(const gpulinear&) = delete;


public:

    //- Runtime type information
    TypeName("linear");


    // Constructors

        //- Construct from mesh
        gpulinear(const gpufvMesh& mesh)
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}

        //- Construct from Istream
        gpulinear(const gpufvMesh& mesh, Istream&)
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}

        //- Construct from faceFlux and Istream
        gpulinear
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField&,
            Istream&
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh)
        {}


    // Member Functions

        //- Return the interpolation weighting factors
        tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const
        {
            return this->mesh().gpusurfaceInterpolation::weights();
        }
};


template<class Type>
tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
linearInterpolate(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf)
{
    return gpusurfaceInterpolationScheme<Type>::interpolate
    (
        vf,
        vf.mesh().gpusurfaceInterpolation::weights()
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
linearInterpolate(const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>& tvf)
{
    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tinterp =
        linearInterpolate(tvf());
    tvf.clear();
    return tinterp;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
