/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::LimitedScheme

Group
    grpFvLimitedSurfaceInterpolationSchemes

Description
    Class to create NVD/TVD limited weighting-factors.

    The particular differencing scheme class is supplied as a template
    argument, the weight function of which is called by the weight function
    of this class for the internal faces as well as faces of coupled
    patches (e.g. processor-processor patches). The weight function is
    supplied the central-differencing weighting factor, the face-flux, the
    cell and face gradients (from which the normalised variable
    distribution may be created) and the cell centre distance.

    This code organisation is both neat and efficient, allowing for
    convenient implementation of new schemes to run on parallelised cases.

SourceFiles
    LimitedScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuLimitedScheme_H
#define gpuLimitedScheme_H

#include "gpulimitedSurfaceInterpolationScheme.H"
#include "gpuLimitFuncs.H"
#include "gpuNVDTVD.H"
#include "gpuNVDVTVDV.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                        Class LimitedScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type, class Limiter, template<class> class LimitFunc>
class gpuLimitedScheme
:
    public gpulimitedSurfaceInterpolationScheme<Type>,
    public Limiter
{
    // Private Member Functions

        //- Calculate the limiter
        void calcLimiter
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& phi,
            surfaceScalargpuField& limiterField
        ) const;

        //- No copy construct
        gpuLimitedScheme(const gpuLimitedScheme&) = delete;

        //- No copy assignment
        void operator=(const gpuLimitedScheme&) = delete;


public:

    //- Runtime type information
    TypeName("gpuLimitedScheme");

    typedef Limiter LimiterType;

    // Constructors

        //- Construct from mesh and faceFlux and limiter scheme
        gpuLimitedScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            const Limiter& weight
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, faceFlux),
            Limiter(weight)
        {}

        //- Construct from mesh and Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpuLimitedScheme
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, is),
            Limiter(is)
        {}

        //- Construct from mesh, faceFlux and Istream
        gpuLimitedScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& is
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, faceFlux),
            Limiter(is)
        {}


    // Member Functions

        //- Return the interpolation weighting factors
        virtual tmp<surfaceScalargpuField> limiter
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makegpuLimitedSurfaceInterpolationTypeScheme\
(                                                                              \
    SS,                                                                        \
    LIMITER,                                                                   \
    gpuNVDTVD,                                                                    \
    LIMFUNC,                                                                   \
    TYPE                                                                       \
)                                                                              \
                                                                               \
typedef gpuLimitedScheme<TYPE, LIMITER<gpuNVDTVD>, gpulimitFuncs::LIMFUNC>              \
    gpuLimitedScheme##TYPE##LIMITER##gpuNVDTVD##LIMFUNC##_;                          \
defineTemplateTypeNameAndDebugWithName                                         \
    (gpuLimitedScheme##TYPE##LIMITER##gpuNVDTVD##LIMFUNC##_, #SS, 0);                \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable                    \
<gpuLimitedScheme<TYPE, LIMITER<gpuNVDTVD>, gpulimitFuncs::LIMFUNC>>                    \
    add##SS##LIMFUNC##TYPE##MeshConstructorToTable_;                           \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable                \
<gpuLimitedScheme<TYPE, LIMITER<gpuNVDTVD>, gpulimitFuncs::LIMFUNC>>                    \
    add##SS##LIMFUNC##TYPE##MeshFluxConstructorToTable_;                       \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable             \
<gpuLimitedScheme<TYPE, LIMITER<gpuNVDTVD>, gpulimitFuncs::LIMFUNC>>                    \
    add##SS##LIMFUNC##TYPE##MeshConstructorToLimitedTable_;                    \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable         \
<gpuLimitedScheme<TYPE, LIMITER<gpuNVDTVD>, gpulimitFuncs::LIMFUNC>>                    \
    add##SS##LIMFUNC##TYPE##MeshFluxConstructorToLimitedTable_;


#define makegpuLimitedSurfaceInterpolationScheme(SS, LIMITER)                     \
                                                                               \
makegpuLimitedSurfaceInterpolationTypeScheme(SS,LIMITER,gpuNVDTVD,magSqr,scalar)     \
makegpuLimitedSurfaceInterpolationTypeScheme(SS,LIMITER,gpuNVDTVD,magSqr,vector)     \
makegpuLimitedSurfaceInterpolationTypeScheme                                      \
(                                                                              \
    SS,                                                                        \
    LIMITER,                                                                   \
    gpuNVDTVD,                                                                    \
    magSqr,                                                                    \
    sphericalTensor                                                            \
)                                                                              \
makegpuLimitedSurfaceInterpolationTypeScheme(SS,LIMITER,gpuNVDTVD,magSqr,symmTensor)\
makegpuLimitedSurfaceInterpolationTypeScheme(SS,LIMITER,gpuNVDTVD,magSqr,tensor)


#define makegpuLimitedVSurfaceInterpolationScheme(SS, LIMITER)                    \
makegpuLimitedSurfaceInterpolationTypeScheme(SS,LIMITER,gpuNVDVTVDV,null,vector)


#define makegpuLLimitedSurfaceInterpolationTypeScheme\
(                                                                              \
    SS,                                                                        \
    LLIMITER,                                                                  \
    LIMITER,                                                                   \
    gpuNVDTVD,                                                                    \
    LIMFUNC,                                                                   \
    TYPE                                                                       \
)                                                                              \
                                                                               \
typedef gpuLimitedScheme<TYPE, LLIMITER<LIMITER<gpuNVDTVD>>, gpulimitFuncs::LIMFUNC>    \
    gpuLimitedScheme##TYPE##LLIMITER##LIMITER##gpuNVDTVD##LIMFUNC##_;                \
defineTemplateTypeNameAndDebugWithName                                         \
    (gpuLimitedScheme##TYPE##LLIMITER##LIMITER##gpuNVDTVD##LIMFUNC##_, #SS, 0);      \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable                    \
<gpuLimitedScheme<TYPE, LLIMITER<LIMITER<gpuNVDTVD>>, gpulimitFuncs::LIMFUNC>>          \
    add##SS##LIMFUNC##TYPE##MeshConstructorToTable_;                           \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable                \
<gpuLimitedScheme<TYPE, LLIMITER<LIMITER<gpuNVDTVD>>, gpulimitFuncs::LIMFUNC>>          \
    add##SS##LIMFUNC##TYPE##MeshFluxConstructorToTable_;                       \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable             \
<gpuLimitedScheme<TYPE, LLIMITER<LIMITER<gpuNVDTVD>>, gpulimitFuncs::LIMFUNC>>          \
    add##SS##LIMFUNC##TYPE##MeshConstructorToLimitedTable_;                    \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable         \
<gpuLimitedScheme<TYPE, LLIMITER<LIMITER<gpuNVDTVD>>, gpulimitFuncs::LIMFUNC>>          \
    add##SS##LIMFUNC##TYPE##MeshFluxConstructorToLimitedTable_;


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuLimitedScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
