/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpusteadyStateDdtScheme.H"
//#include "fvcDiv.H"
#include "gpufvMatrices.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpusteadyStateDdtScheme<Type>::fvcDdt
(
    const dimensioned<Type>& dt
)
{
    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        IOobject
        (
            "ddt("+dt.name()+')',
            mesh().time().timeName(),
            mesh().hostmesh()
        ),
        mesh(),
        dimensioned<Type>(dt.dimensions()/dimTime, Zero)
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpusteadyStateDdtScheme<Type>::fvcDdt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        IOobject
        (
            "ddt("+vf.name()+')',
            mesh().time().timeName(),
            mesh().hostmesh()
        ),
        mesh(),
        dimensioned<Type>(vf.dimensions()/dimTime, Zero)
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpusteadyStateDdtScheme<Type>::fvcDdt
(
    const dimensionedScalar& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        IOobject
        (
            "ddt("+rho.name()+','+vf.name()+')',
            mesh().time().timeName(),
            mesh().hostmesh()
        ),
        mesh(),
        dimensioned<Type>(rho.dimensions()*vf.dimensions()/dimTime, Zero)
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpusteadyStateDdtScheme<Type>::fvcDdt
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        IOobject
        (
            "ddt("+rho.name()+','+vf.name()+')',
            mesh().time().timeName(),
            mesh().hostmesh()
        ),
        mesh(),
        dimensioned<Type>(rho.dimensions()*vf.dimensions()/dimTime, Zero)
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpusteadyStateDdtScheme<Type>::fvcDdt
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        IOobject
        (
            "ddt("+alpha.name()+','+rho.name()+','+vf.name()+')',
            mesh().time().timeName(),
            mesh().hostmesh()
        ),
        mesh(),
        dimensioned<Type>(rho.dimensions()*vf.dimensions()/dimTime, Zero)
    );
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpusteadyStateDdtScheme<Type>::fvmDdt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            vf.dimensions()*dimVol/dimTime
        )
    );

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpusteadyStateDdtScheme<Type>::fvmDdt
(
    const dimensionedScalar& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpusteadyStateDdtScheme<Type>::fvmDdt
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpusteadyStateDdtScheme<Type>::fvmDdt
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            alpha.dimensions()*rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );

    return tfvm;
}


template<class Type>
tmp<typename gpusteadyStateDdtScheme<Type>::fluxFieldType>
gpusteadyStateDdtScheme<Type>::fvcDdtUfCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    tmp<fluxFieldType> tCorr
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr(" + U.name() + ',' + Uf.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            mesh(),
            dimensioned<typename flux<Type>::type>
            (
                Uf.dimensions()*dimArea/dimTime, Zero
            )
        )
    );

    tCorr.ref().setOriented();

    return tCorr;
}


template<class Type>
tmp<typename gpusteadyStateDdtScheme<Type>::fluxFieldType>
gpusteadyStateDdtScheme<Type>::fvcDdtPhiCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const fluxFieldType& phi
)
{
    tmp<fluxFieldType> tCorr
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr(" + U.name() + ',' + phi.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            mesh(),
            dimensioned<typename flux<Type>::type>
            (
                phi.dimensions()/dimTime, Zero
            )
        )
    );

    tCorr.ref().setOriented();

    return tCorr;
}


template<class Type>
tmp<typename gpusteadyStateDdtScheme<Type>::fluxFieldType>
gpusteadyStateDdtScheme<Type>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    tmp<fluxFieldType> tCorr
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr("
              + rho.name()
              + ',' + U.name() + ',' + Uf.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            mesh(),
            dimensioned<typename flux<Type>::type>
            (
                Uf.dimensions()*dimArea/dimTime, Zero
            )
        )
    );

    tCorr.ref().setOriented();

    return tCorr;
}


template<class Type>
tmp<typename gpusteadyStateDdtScheme<Type>::fluxFieldType>
gpusteadyStateDdtScheme<Type>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const fluxFieldType& phi
)
{
    tmp<fluxFieldType> tCorr
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr("
              + rho.name()
              + ',' + U.name() + ',' + phi.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            mesh(),
            dimensioned<typename flux<Type>::type>
            (
                phi.dimensions()/dimTime, Zero
            )
        )
    );

    tCorr.ref().setOriented();

    return tCorr;
}


template<class Type>
tmp<surfaceScalargpuField> gpusteadyStateDdtScheme<Type>::meshPhi
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return tmp<surfaceScalargpuField>::New
    (
        IOobject
        (
            "meshPhi",
            mesh().time().timeName(),
            mesh().hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        mesh(),
        dimensionedScalar(dimVolume/dimTime, Zero)
    );
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// ************************************************************************* //
