/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2014 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::processorLduInterfaceField

Description
    Abstract base class for processor coupled interfaces.

SourceFiles
    processorLduInterfaceField.C

\*---------------------------------------------------------------------------*/

#ifndef processorLduInterfacegpuField_H
#define processorLduInterfacegpuField_H

#include "primitiveFieldsFwd.H"
#include "typeInfo.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class processorLduInterfaceField Declaration
\*---------------------------------------------------------------------------*/

class processorLduInterfacegpuField
{

public:

    //- Runtime type information
    TypeName("processorLduInterfacegpuField");


    // Constructors

        //- Construct null
        processorLduInterfacegpuField() = default;


    //- Destructor
    virtual ~processorLduInterfacegpuField() = default;


    // Member Functions

        // Access

            //- Return communicator used for comms
            virtual label comm() const = 0;

            //- Return processor number
            virtual int myProcNo() const = 0;

            //- Return neighbour processor number
            virtual int neighbProcNo() const = 0;

            //- Is the transform required
            virtual bool doTransform() const = 0;

            //- Return face transformation tensor
            virtual const tensorgpuField& gpuForwardT() const = 0;

            //- Return rank of component for transform
            virtual int rank() const = 0;


        //- Transform given patch field

        template<class Type>
        void transformCoupleField(gpuField<Type>& f) const;

        //- Transform given patch component field

        void transformCoupleField
        (
            scalargpuField& f,
            const direction cmpt
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "tensorField.H"


template<class Type>
void Foam::processorLduInterfacegpuField::transformCoupleField
(
    gpuField<Type>& f
) const
{
    if (doTransform())
    {
        if (gpuForwardT().size() == 1)
        {
            transform(f, gpuForwardT().first(), f);
        }
        else
        {
            transform(f, gpuForwardT(), f);
        }
    }
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
