/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::linearUpwindV

Description
    linearUpwindV interpolation scheme class derived from upwind and returns
    upwind weighting factors but also applies an explicit correction.

SourceFiles
    linearUpwindV.C

\*---------------------------------------------------------------------------*/

#ifndef gpulinearUpwindV_H
#define gpulinearUpwindV_H

#include "gpuupwind.H"
#include "gpugaussGrad.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class linearUpwindV Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulinearUpwindV
:
    public gpuupwind<Type>
{
    // Private Data

        word gradSchemeName_;
        tmp<fv::gpugradScheme<Type>> gradScheme_;


    // Private Member Functions

        //- No copy construct
        gpulinearUpwindV(const gpulinearUpwindV&) = delete;

        //- No copy assignment
        void operator=(const gpulinearUpwindV&) = delete;


public:

    //- Runtime type information
    TypeName("linearUpwindV");


    // Constructors

        //- Construct from faceFlux
        gpulinearUpwindV
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux
        )
        :
            gpuupwind<Type>(mesh, faceFlux),
            gradSchemeName_("grad"),
            gradScheme_
            (
                new fv::gpugaussGrad<Type>(mesh)
            )
        {}

        //- Construct from Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpulinearUpwindV
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        )
        :
            gpuupwind<Type>(mesh, schemeData),
            gradSchemeName_(schemeData),
            gradScheme_
            (
                fv::gpugradScheme<Type>::New
                (
                    mesh,
                    mesh.hostmesh().gradScheme(gradSchemeName_)
                )
            )
        {}

        //- Construct from faceFlux and Istream
        gpulinearUpwindV
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        )
        :
            gpuupwind<Type>(mesh, faceFlux, schemeData),
            gradSchemeName_(schemeData),
            gradScheme_
            (
                fv::gpugradScheme<Type>::New
                (
                    mesh,
                    mesh.hostmesh().gradScheme(gradSchemeName_)
                )
            )
        {}


    // Member Functions

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const
        {
            return true;
        }

        //- Return the explicit correction to the face-interpolate
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
