/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2015 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::processorFvPatch

Description
    Processor patch.

SourceFiles
    processorFvPatch.C

\*---------------------------------------------------------------------------*/

#ifndef processorgpuFvPatch_H
#define processorgpuFvPatch_H

#include "coupledgpuFvPatch.H"
#include "processorgpuLduInterface.H"
#include "processorPolyPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class processorFvPatch Declaration
\*---------------------------------------------------------------------------*/

class processorgpuFvPatch
:
    public coupledgpuFvPatch,
    public processorgpuLduInterface
{
    // Private Data

        const processorPolyPatch& procPolyPatch_;


protected:

    // Protected Member functions

        //- Make patch weighting factors
        void makeWeights(scalargpuField&) const;


public:

    //- Runtime type information
    TypeName(processorPolyPatch::typeName_());


    // Constructors

        //- Construct from components
        processorgpuFvPatch(const polyPatch& patch, const gpufvBoundaryMesh& bm)
        :
            coupledgpuFvPatch(patch, bm),
            procPolyPatch_(refCast<const processorPolyPatch>(patch))
        {}


    // Member functions

        //- Return communicator used for comms
        virtual label comm() const
        {
            return procPolyPatch_.comm();
        }

        //- Return processor number
        virtual int myProcNo() const
        {
            return procPolyPatch_.myProcNo();
        }

        //- Return neighbour processor number
        virtual int neighbProcNo() const
        {
            return procPolyPatch_.neighbProcNo();
        }

        //- Return message tag used for sending
        virtual int tag() const
        {
            return UPstream::msgType();
        }

        //- Return true if running parallel
        virtual bool coupled() const
        {
            return Pstream::parRun();
        }

        const processorPolyPatch& procPolyPatch() const
        {
            return procPolyPatch_;
        }

        //- Are the cyclic planes parallel
        virtual bool parallel() const
        {
            return procPolyPatch_.parallel();
        }

        //- Return face transformation tensor
        virtual const tensorgpuField& gpuForwardT() const
        {
            return procPolyPatch_.gpuForwardT();
        }

        //- Return neighbour-cell transformation tensor.
        virtual const tensorgpuField& gpuReverseT() const
        {
            return procPolyPatch_.gpuReverseT();
        }

        //- Return delta (P to N) vectors across coupled patch
        virtual tmp<vectorgpuField> delta() const;


        // Interface transfer functions

            //- Return the values of the given internal data adjacent to
            //- the interface as a field
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData
            ) const;

            //- Return the values of the given internal data adjacent to
            //- the interface as a field using a mapping faceCell
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData,
                const labelgpuList& faceCells
            ) const;

            //- Inherit initInternalFieldTransfer from coupledFvPatch
            using coupledgpuFvPatch::initInternalFieldTransfer;

            //- Initialise neighbour field transfer
            virtual void initInternalFieldTransfer
            (
                const Pstream::commsTypes commsType,
                const labelgpuList& internalData
            ) const;

            //- Initialise neighbour field transfer using faceCells map
            virtual void initInternalFieldTransfer
            (
                const Pstream::commsTypes commsType,
                const labelgpuList& iF,
                const labelgpuList& faceCells
            ) const;

            //- Return neighbour field
            virtual tmp<labelgpuField> internalFieldTransfer
            (
                const Pstream::commsTypes commsType,
                const labelgpuList& internalData
            ) const;

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
