/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicAMIFvPatch

Description
    Cyclic patch for Arbitrary Mesh Interface (AMI)

SourceFiles
    cyclicAMIFvPatch.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicAMIgpuFvPatch_H
#define cyclicAMIgpuFvPatch_H

#include "coupledgpuFvPatch.H"
#include "cyclicAMIgpuLduInterface.H"
#include "cyclicAMIPolyPatch.H"
#include "gpufvBoundaryMesh.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class cyclicAMIFvPatch Declaration
\*---------------------------------------------------------------------------*/

class cyclicAMIgpuFvPatch
:
    public coupledgpuFvPatch,
    public cyclicAMIgpuLduInterface
{
    // Private data

        const cyclicAMIPolyPatch& cyclicAMIPolyPatch_;


protected:

    // Protected Member functions

        //- Make patch weighting factors
        void makeWeights(scalargpuField&) const;

        //- Correct patch deltaCoeffs
        virtual void makeDeltaCoeffs(scalargpuField&) const;

        //- Correct patch non-ortho deltaCoeffs
        virtual void makeNonOrthoDeltaCoeffs(scalargpuField&) const;

        //- Correct patch non-ortho correction vectors
        virtual void makeNonOrthoCorrVectors(vectorgpuField&) const;

        //- Correct patches after moving points
        virtual void movePoints();


public:

    //- Runtime type information
    TypeName(cyclicAMIPolyPatch::typeName_());


    // Constructors

        //- Construct from polyPatch
        cyclicAMIgpuFvPatch(const polyPatch& patch, const gpufvBoundaryMesh& bm)
        :
            coupledgpuFvPatch(patch, bm),
            cyclicAMIgpuLduInterface(),
            cyclicAMIPolyPatch_(refCast<const cyclicAMIPolyPatch>(patch))
        {}


    // Member functions

        // Access

//             // Implicit treatment functions
//
//                 //- Return number of new internal of this polyPatch faces
//                 virtual void newInternalProcFaces(label&, label&) const;
//
//                 //- Return nbrCells
//                 virtual const labelUList& nbrCells() const
//                 {
//                     return cyclicAMIPolyPatch_.neighbPatch().faceCells();
//                 }
//
//                 //- Return nbr patch ID
//                 virtual label neighbPolyPatchID() const
//                 {
//                     return neighbPatchID();
//                 }
//
//                 //- Return collocated faces map
//                 virtual refPtr<labelListList> mapCollocatedFaces() const
//                 {
//                     const labelListList& sourceFaces =
//                         cyclicAMIPolyPatch_.AMI().srcAddress();
//                     return refPtr<labelListList>
//                     (
//                         new labelListList(sourceFaces)
//                     );
//                 }
//
//                 //- Return implicit master
//                 virtual bool masterImplicit() const
//                 {
//                     return owner();
//                 }


            //- Return local reference cast into the cyclic patch
            const cyclicAMIPolyPatch& cyclicAMIPatch() const
            {
                return cyclicAMIPolyPatch_;
            }

            //- Return neighbour
            virtual label neighbPatchID() const
            {
                return cyclicAMIPolyPatch_.neighbPatchID();
            }

            virtual bool owner() const
            {
                return cyclicAMIPolyPatch_.owner();
            }

            //- Return processor number
            virtual const cyclicAMIgpuFvPatch& neighbPatch() const
            {
                return refCast<const cyclicAMIgpuFvPatch>
                (
                    this->boundaryMesh()[cyclicAMIPolyPatch_.neighbPatchID()]
                );
            }

            //- Return a reference to the AMI interpolator
            virtual const AMIPatchToPatchInterpolation& AMI() const
            {
                return cyclicAMIPolyPatch_.AMI();
            }

            //- Return true if applying the low weight correction
            virtual bool applyLowWeightCorrection() const
            {
                return cyclicAMIPolyPatch_.applyLowWeightCorrection();
            }


            //- Are the cyclic planes parallel
            virtual bool parallel() const
            {
                return cyclicAMIPolyPatch_.parallel();
            }

            //- Return face transformation tensor

            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicAMIPolyPatch_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor

            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicAMIPolyPatch_.gpuReverseT();
            }

            const cyclicAMIgpuFvPatch& neighbFvPatch() const
            {
                return refCast<const cyclicAMIgpuFvPatch>
                (
                    this->boundaryMesh()[cyclicAMIPolyPatch_.neighbPatchID()]
                );
            }


            //- Return true if this patch is coupled. This is equivalent
            //- to the coupledPolyPatch::coupled() if parallel running or
            //- both sides present, false otherwise
            virtual bool coupled() const;

            //- Return delta (P to N) vectors across coupled patch
            virtual tmp<vectorgpuField> delta() const;

            template<class Type>
            tmp<Field<Type>> interpolate
            (
                const Field<Type>& fld,
                const UList<Type>& defaultValues = UList<Type>()
            ) const
            {
                return cyclicAMIPolyPatch_.interpolate(fld, defaultValues);
            }

            template<class Type>
            tmp<Field<Type>> interpolate
            (
                const tmp<Field<Type>>& tFld,
                const UList<Type>& defaultValues = UList<Type>()
            ) const
            {
                return cyclicAMIPolyPatch_.interpolate(tFld, defaultValues);
            }

            template<class Type>
            tmp<gpuField<Type>> interpolate
            (
                const gpuField<Type>& fld,
                const gpuList<Type>& defaultValues = gpuList<Type>()
            ) const
            {
                return cyclicAMIPolyPatch_.interpolate(fld, defaultValues);
            }

            template<class Type>
            tmp<gpuField<Type>> interpolate
            (
                const tmp<gpuField<Type>>& tFld,
                const gpuList<Type>& defaultValues = gpuList<Type>()
            ) const
            {
                return cyclicAMIPolyPatch_.interpolate(tFld, defaultValues);
            }

        // Interface transfer functions

            //- Return the values of the given internal data adjacent to
            //- the interface as a field
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData
            ) const;


            //- Return the values of the given internal data adjacent to
            //- the interface as a field using a mapping faceCell
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData,
                const labelgpuList& faceCells
            ) const;


            //- Return neighbour field
            virtual tmp<labelgpuField> internalFieldTransfer
            (
                const Pstream::commsTypes commsType,
                const labelgpuList& internalData
            ) const;

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
