/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "cyclicACMIgpuFvPatch.H"
#include "gpufvMesh.H"
#include "transform.H"
#include "surfacegpuFields.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(cyclicACMIgpuFvPatch, 0);
    addToRunTimeSelectionTable(gpufvPatch, cyclicACMIgpuFvPatch, polyPatch);

    struct cyclicACMIFvPatchMakeWeightsFunctor
    {
        __host__ __device__
        scalar operator () (const scalar& deltas, const scalar& nbrDeltas)
        {
            return nbrDeltas/(deltas + nbrDeltas);
        }
    };
}


// * * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * //

bool Foam::cyclicACMIgpuFvPatch::updateAreas() const
{
    // Give AMI chance to update itself
    bool updated = cyclicACMIPolyPatch_.updateAreas();

    if (!cyclicACMIPolyPatch_.owner())
    {
        return updated;
    }

    if (updated || !cyclicACMIPolyPatch_.upToDate(areaTime_))
    {
        if (debug)
        {
            Pout<< "cyclicACMIFvPatch::updateAreas() : updating fv areas for "
                << name() << " and " << this->nonOverlapPatch().name()
                << endl;
        }

        const gpufvPatch& nonOverlapPatch = this->nonOverlapPatch();
        const cyclicACMIgpuFvPatch& nbrACMI = neighbPatch();
        const gpufvPatch& nbrNonOverlapPatch = nbrACMI.nonOverlapPatch();

        resetPatchAreas(*this);
        resetPatchAreas(nonOverlapPatch);
        resetPatchAreas(nbrACMI);
        resetPatchAreas(nbrNonOverlapPatch);

        updated = true;

        // Mark my data to be up to date with ACMI polyPatch level
        cyclicACMIPolyPatch_.setUpToDate(areaTime_);
    }
    return updated;
}


void Foam::cyclicACMIgpuFvPatch::resetPatchAreas(const gpufvPatch& fvp) const
{
    const_cast<vectorgpuField&>(fvp.Sf()) = fvp.patch().faceAreas();
    const_cast<vectorgpuField&>(fvp.Cf()) = fvp.patch().faceCentres();
    const_cast<scalargpuField&>(fvp.magSf()) = mag(fvp.patch().faceAreas());

    DebugPout
        << fvp.patch().name() << " area:" << sum(fvp.magSf()) << endl;
}


void Foam::cyclicACMIgpuFvPatch::makeWeights(scalargpuField& w) const
{
    if (coupled())
    {
        const cyclicACMIgpuFvPatch& nbrPatch = neighbFvPatch();
        const scalargpuField deltas(nf() & coupledgpuFvPatch::delta());

        // These deltas are of the cyclic part alone - they are
        // not affected by the amount of overlap with the nonOverlapPatch
        scalargpuField nbrDeltas
        (
            interpolate
            (
                nbrPatch.nf() & nbrPatch.coupledgpuFvPatch::delta()
            )
        );

        //const scalar tol = cyclicACMIPolyPatch::tolerance();
        //if (dni < tol)  w[facei] = 1.0;  need to do!

        thrust::transform
        (
            deltas.begin(),
            deltas.end(),
            nbrDeltas.begin(),
            w.begin(),
            cyclicACMIFvPatchMakeWeightsFunctor()
        );
    }
    else
    {
        // Behave as uncoupled patch
        gpufvPatch::makeWeights(w);
    }
}


// * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * * * //

Foam::cyclicACMIgpuFvPatch::cyclicACMIgpuFvPatch
(
    const polyPatch& patch,
    const gpufvBoundaryMesh& bm
)
:
    coupledgpuFvPatch(patch, bm),
    cyclicACMIgpuLduInterface(),
    cyclicACMIPolyPatch_(refCast<const cyclicACMIPolyPatch>(patch)),
    areaTime_
    (
        IOobject
        (
            "areaTime",
            boundaryMesh().mesh().hostmesh().pointsInstance(),
            boundaryMesh().mesh().hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        dimensionedScalar("time", dimTime, -GREAT)
    )
{
    areaTime_.eventNo() = -1;
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

// void Foam::cyclicACMIFvPatch::newInternalProcFaces
// (
//     label& newFaces,
//     label& newProcFaces
// ) const
// {
//     const List<labelList>& addSourceFaces =
//         cyclicACMIPolyPatch_.AMI().srcAddress();
//
//     const scalarField& fMask = cyclicACMIPolyPatch_.srcMask();
//
//     // Add new faces as many weights for AMI
//     forAll (addSourceFaces, faceI)
//     {
//         if (fMask[faceI] > cyclicACMIPolyPatch_.tolerance_)
//         {
//             const labelList& nbrFaceIs = addSourceFaces[faceI];
//
//             forAll (nbrFaceIs, j)
//             {
//                 label nbrFaceI = nbrFaceIs[j];
//
//                 if (nbrFaceI < neighbPatch().size())
//                 {
//                     // local faces
//                     newFaces++;
//                 }
//                 else
//                 {
//                     // Proc faces
//                     newProcFaces++;
//                 }
//             }
//         }
//     }
// }


// Foam::refPtr<Foam::labelListList>
// Foam::cyclicACMIFvPatch::mapCollocatedFaces() const
// {
//     const scalarField& fMask = cyclicACMIPolyPatch_.srcMask();
//     const labelListList& srcFaces = cyclicACMIPolyPatch_.AMI().srcAddress();
//     labelListList dOverFaces;
//
//     dOverFaces.setSize(srcFaces.size());
//     forAll (dOverFaces, faceI)
//     {
//         if (fMask[faceI] > cyclicACMIPolyPatch_.tolerance_)
//         {
//             dOverFaces[faceI].setSize(srcFaces[faceI].size());
//
//             forAll (dOverFaces[faceI], subFaceI)
//             {
//                 dOverFaces[faceI][subFaceI] = srcFaces[faceI][subFaceI];
//             }
//         }
//     }
//     return refPtr<labelListList>(new labelListList(dOverFaces));
// }


bool Foam::cyclicACMIgpuFvPatch::coupled() const
{
    return Pstream::parRun() || (this->size() && neighbFvPatch().size());
}


Foam::tmp<Foam::vectorgpuField> Foam::cyclicACMIgpuFvPatch::delta() const
{
    if (coupled())
    {
        const cyclicACMIgpuFvPatch& nbrPatch = neighbFvPatch();

        const vectorgpuField patchD(coupledgpuFvPatch::delta());

        vectorgpuField nbrPatchD(interpolate(nbrPatch.coupledgpuFvPatch::delta()));

        auto tpdv = tmp<vectorgpuField>::New(patchD.size());
        vectorgpuField& pdv = tpdv.ref();

        // do the transformation if necessary
        if (parallel())
        {
            thrust::transform
            (
                patchD.begin(),
                patchD.end(),
                nbrPatchD.begin(),
                pdv.begin(),
                subtractOperatorFunctor<vector,vector,vector>()
            );
        }
        else
        {
            tensor t = gpuForwardT().first();
			
            thrust::transform
            (
                patchD.begin(),
                patchD.end(),
                thrust::make_transform_iterator
                (
                    nbrPatchD.begin(),
                    transformBinaryFunctionSFFunctor<tensor,vector,vector>(t)
                ),
                pdv.begin(),
                subtractOperatorFunctor<vector,vector,vector>()
            );
        }

        return tpdv;
    }
    else
    {
        return coupledgpuFvPatch::delta();
    }
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicACMIgpuFvPatch::interfaceInternalField
(
    const labelgpuList& internalData
) const
{
    return patchInternalField(internalData);
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicACMIgpuFvPatch::interfaceInternalField
(
    const labelgpuList& internalData,
    const labelgpuList& faceCells
) const
{
    return patchInternalField(internalData, faceCells);
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicACMIgpuFvPatch::internalFieldTransfer
(
    const Pstream::commsTypes commsType,
    const labelgpuList& iF
) const
{
    return neighbFvPatch().patchInternalField(iF);
}


void Foam::cyclicACMIgpuFvPatch::movePoints()
{
/*
    if (!cyclicACMIPolyPatch_.owner())
    {
        return;
    }


    if (!cyclicACMIPolyPatch_.upToDate(areaTime_))
    {
        if (debug)
        {
            Pout<< "cyclicACMIFvPatch::movePoints() : updating fv areas for "
                << name() << " and " << this->nonOverlapPatch().name()
                << endl;
        }


        // Set the patch face areas to be consistent with the changes made
        // at the polyPatch level

        const fvPatch& nonOverlapPatch = this->nonOverlapPatch();
        const cyclicACMIFvPatch& nbrACMI = neighbPatch();
        const fvPatch& nbrNonOverlapPatch = nbrACMI.nonOverlapPatch();

        resetPatchAreas(*this);
        resetPatchAreas(nonOverlapPatch);
        resetPatchAreas(nbrACMI);
        resetPatchAreas(nbrNonOverlapPatch);

        // Scale the mesh flux

        const labelListList& newSrcAddr = AMI().srcAddress();
        const labelListList& newTgtAddr = AMI().tgtAddress();

        const fvMesh& mesh = boundaryMesh().mesh();
        surfaceScalarField& meshPhi = const_cast<fvMesh&>(mesh).setPhi();
        surfaceScalarField::Boundary& meshPhiBf = meshPhi.boundaryFieldRef();

        // Note: phip and phiNonOverlap will be different sizes if new faces
        // have been added
        scalarField& phip = meshPhiBf[cyclicACMIPolyPatch_.index()];
        scalarField& phiNonOverlapp =
            meshPhiBf[nonOverlapPatch.patch().index()];

        const auto& points = mesh.points();

        forAll(phip, facei)
        {
            if (newSrcAddr[facei].empty())
            {
                // AMI patch with no connection to other coupled faces
                phip[facei] = 0.0;
            }
            else
            {
                // Scale the mesh flux according to the area fraction
                const face& fAMI = cyclicACMIPolyPatch_[facei];

                // Note: using raw point locations to calculate the geometric
                // area - faces areas are currently scaled (decoupled from
                // mesh points)
                const scalar geomArea = fAMI.mag(points);
                phip[facei] *= magSf()[facei]/geomArea;
            }
        }

        forAll(phiNonOverlapp, facei)
        {
            const scalar w = 1.0 - cyclicACMIPolyPatch_.srcMask()[facei];
            phiNonOverlapp[facei] *= w;
        }

        const cyclicACMIPolyPatch& nbrPatch = nbrACMI.cyclicACMIPatch();
        scalarField& nbrPhip = meshPhiBf[nbrPatch.index()];
        scalarField& nbrPhiNonOverlapp =
            meshPhiBf[nbrNonOverlapPatch.patch().index()];

        forAll(nbrPhip, facei)
        {
            if (newTgtAddr[facei].empty())
            {
                nbrPhip[facei] = 0.0;
            }
            else
            {
                const face& fAMI = nbrPatch[facei];

                // Note: using raw point locations to calculate the geometric
                // area - faces areas are currently scaled (decoupled from
                // mesh points)
                const scalar geomArea = fAMI.mag(points);
                nbrPhip[facei] *= nbrACMI.magSf()[facei]/geomArea;
            }
        }

        forAll(nbrPhiNonOverlapp, facei)
        {
            const scalar w = 1.0 - cyclicACMIPolyPatch_.tgtMask()[facei];
            nbrPhiNonOverlapp[facei] *= w;
        }

        // Mark my data to be up to date with ACMI polyPatch level
        cyclicACMIPolyPatch_.setUpToDate(areaTime_);
    }
*/
}

// ************************************************************************* //
