/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2018-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvBoundaryMesh

Description
    Foam::fvBoundaryMesh

SourceFiles
    fvBoundaryMesh.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvBoundaryMesh_H
#define gpufvBoundaryMesh_H

#include "gpufvPatchList.H"
#include "gpulduInterfacePtrsList.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward Declarations
class gpufvMesh;
class polyBoundaryMesh;
class wordRes;

/*---------------------------------------------------------------------------*\
                       Class fvBoundaryMesh Declaration
\*---------------------------------------------------------------------------*/

class gpufvBoundaryMesh
:
    public gpufvPatchList
{
    // Private Data

        //- Reference to mesh
        const gpufvMesh& mesh_;


    // Private Member Functions

        //- Add fvPatches corresponding to the given polyBoundaryMesh
        void addPatches(const polyBoundaryMesh&);


        //- No copy construct
        gpufvBoundaryMesh(const gpufvBoundaryMesh&) = delete;

        //- No copy assignment
        void operator=(const gpufvBoundaryMesh&) = delete;


protected:



public:

    friend class gpufvMesh;


    // Constructors

        //- Construct zero size with mesh reference
        explicit gpufvBoundaryMesh(const gpufvMesh&);

        //- Construct from polyBoundaryMesh
        gpufvBoundaryMesh(const gpufvMesh&, const polyBoundaryMesh&);


    // Member Functions

        //- Return the mesh reference
        const gpufvMesh& mesh() const noexcept
        {
            return mesh_;
        }

        //- Return a list of faceCells for each patch
        UPtrList<const labelgpuList> gpuFaceCells() const;

        //- Return a list of pointers for each patch
        //- with only those pointing to interfaces being set
        gpulduInterfacePtrsList interfaces() const;

        //- Return (sorted) patch indices for all matches.
        //  A no-op (returns empty list) for an empty matcher
        labelList indices(const wordRe& matcher, const bool useGroups) const;

        //- Return (sorted) patch indices for all matches.
        //  A no-op (returns empty list) for an empty matcher
        labelList indices(const wordRes& matcher, const bool useGroups) const;

        //- Find patch index given a name
        //  A no-op (returns -1) for an empty patchName
        label findPatchID(const word& patchName) const;



    // Member Operators

        //- Return const and non-const reference to fvPatch by index.
        using gpufvPatchList::operator[];

        //- Return const reference to fvPatch by name.
        const gpufvPatch& operator[](const word&) const;

        //- Return reference to fvPatch by name.
        gpufvPatch& operator[](const word&);


    // Housekeeping

        //- Identical to the indices() method (AUG-2018)
        FOAM_DEPRECATED_FOR(2018-08, "indices() method")
        labelList findIndices(const wordRe& key, bool useGroups) const
        {
            return indices(key, useGroups);
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
