/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2016-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "LduMatrix.H"
#include "diagTensorField.H"
#include "profiling.H"
#include "PrecisionAdaptor.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::gpufvMatrix<Type>::setComponentReference
(
    const label patchi,
    const label facei,
    const direction cmpt,
    const scalar value
)
{
    if (psi_.needReference())
    {
        if (Pstream::master())
        {
            scalar delta = gpuDiag().get(psi_.mesh().boundary()[patchi].gpuFaceCells().get(facei));
            internalCoeffs_[patchi].set(facei, internalCoeffs_[patchi].get(facei)+delta);
            boundaryCoeffs_[patchi].set(facei,boundaryCoeffs_[patchi].get(facei)+delta*value); 
        }
    }
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::solveSegregatedOrCoupled
(
    const dictionary& solverControls
)
{
    word regionName;
    if (psi_.mesh().name() != polyMesh::defaultRegion)
    {
        regionName = psi_.mesh().name() + "::";
    }
    addProfiling(solve, "gpufvMatrix::solve." + regionName + psi_.name());

    if (debug)
    {
        Info.masterStream(this->mesh().comm())
            << "gpufvMatrix<Type>::solveSegregatedOrCoupled"
               "(const dictionary& solverControls) : "
               "solving gpufvMatrix<Type>"
            << endl;
    }

    // Do not solve if maxIter == 0
    if (solverControls.getOrDefault<label>("maxIter", -1) == 0)
    {
        return SolverPerformance<Type>();
    }

    word type(solverControls.getOrDefault<word>("type", "segregated"));

    if (type == "segregated")
    {
        return solveSegregated(solverControls);
    }
    else if (type == "coupled")
    {
        return solveCoupled(solverControls);
    }
    else
    {
        FatalIOErrorInFunction(solverControls)
            << "Unknown type " << type
            << "; currently supported solver types are segregated and coupled"
            << exit(FatalIOError);

        return SolverPerformance<Type>();
    }
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::solveSegregated
(
    const dictionary& solverControls
)
{
    if (useImplicit_)
    {
        FatalErrorInFunction
            << "Implicit option is not allowed for type: " << Type::typeName
            << exit(FatalError);
    }

    if (debug)
    {
        Info.masterStream(this->mesh().comm())
            << "gpufvMatrix<Type>::solveSegregated"
               "(const dictionary& solverControls) : "
               "solving gpufvMatrix<Type>"
            << endl;
    }

    const int logLevel =
        solverControls.getOrDefault<int>
        (
            "log",
            SolverPerformance<Type>::debug
        );

    auto& psi =
        const_cast<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&>(psi_);

    SolverPerformance<Type> solverPerfVec
    (
        "gpufvMatrix<Type>::solveSegregated",
        psi.name()
    );

    scalargpuField saveDiag(gpuDiag());

    gpuField<Type> source(source_);

    // At this point include the boundary source from the coupled boundaries.
    // This is corrected for the implicit part by updateMatrixInterfaces within
    // the component loop.
    addBoundarySource(source);

    typename Type::labelType validComponents
    (
        psi.mesh().hostmesh().template validComponents<Type>()
    );

    for (direction cmpt=0; cmpt<Type::nComponents; cmpt++)
    {
        if (validComponents[cmpt] == -1) continue;

        // copy field and source

        scalargpuField psiCmpt(psi.primitiveField().component(cmpt));
        addBoundaryDiag(gpuDiag(), cmpt);

        scalargpuField sourceCmpt(source.component(cmpt));

        FieldField<gpuField, scalar> bouCoeffsCmpt
        (
            boundaryCoeffs_.component(cmpt)
        );

        FieldField<gpuField, scalar> intCoeffsCmpt
        (
            internalCoeffs_.component(cmpt)
        );

        lduInterfacegpuFieldPtrsList interfaces =
            psi.boundaryField().scalarInterfaces();

        // Use the initMatrixInterfaces and updateMatrixInterfaces to correct
        // bouCoeffsCmpt for the explicit part of the coupled boundary
        // conditions
        {
            const label startRequest = Pstream::nRequests();

            initMatrixInterfaces
            (
                true,
                bouCoeffsCmpt,
                interfaces,
                psiCmpt,
                sourceCmpt,
                cmpt
            );

            updateMatrixInterfaces
            (
                true,
                bouCoeffsCmpt,
                interfaces,
                psiCmpt,
                sourceCmpt,
                cmpt,
                startRequest
            );
        }

        solverPerformance solverPerf;

        // Solver call
        solverPerf = gpulduMatrix::solver::New
        (
            psi.name() + pTraits<Type>::componentNames[cmpt],
            *this,
            bouCoeffsCmpt,
            intCoeffsCmpt,
            interfaces,
            solverControls
        )->solve(psiCmpt, sourceCmpt, cmpt);

        if (logLevel)
        {
            solverPerf.print(Info.masterStream(this->mesh().comm()));
        }

        solverPerfVec.replace(cmpt, solverPerf);
        solverPerfVec.solverName() = solverPerf.solverName();

        psi.primitiveFieldRef().replace(cmpt, psiCmpt);
        gpuDiag() = saveDiag;
    }

    psi.correctBoundaryConditions();

    psi.mesh().hostmesh().setSolverPerformance(psi.name(), solverPerfVec);

    return solverPerfVec;
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::solveCoupled
(
    const dictionary& solverControls
)
{
/*
    if (debug)
    {
        Info.masterStream(this->mesh().comm())
            << "fvMatrix<Type>::solveCoupled"
               "(const dictionary& solverControls) : "
               "solving fvMatrix<Type>"
            << endl;
    }

    const int logLevel =
        solverControls.getOrDefault<int>
        (
            "log",
            SolverPerformance<Type>::debug
        );

    auto& psi =
        const_cast<GeometricField<Type, fvPatchField, volMesh>&>(psi_);

    LduMatrix<Type, scalar, scalar> coupledMatrix(psi.mesh());

    coupledMatrix.diag() = gpuDiag().asField();
    coupledMatrix.upper() = gpuUpper().asField();
    coupledMatrix.lower() = gpuLower().asField();
    coupledMatrix.source() = source().asField();

    addBoundaryDiag(coupledMatrix.diag(), 0);
    addBoundarySource(coupledMatrix.source(), false);

    coupledMatrix.interfaces() = psi.boundaryFieldRef().interfaces();
    coupledMatrix.interfacesUpper() = boundaryCoeffs().component(0);
    coupledMatrix.interfacesLower() = internalCoeffs().component(0);

    autoPtr<typename LduMatrix<Type, scalar, scalar>::solver>

    coupledMatrixSolver
    (
        LduMatrix<Type, scalar, scalar>::solver::New
        (
            psi.name(),
            coupledMatrix,
            solverControls
        )
    );

    SolverPerformance<Type> solverPerf
    (
        coupledMatrixSolver->solve(psi.getField().asField())
    );

    if (logLevel)
    {
        solverPerf.print(Info.masterStream(this->mesh().comm()));
    }

    psi.correctBoundaryConditions();

    psi.mesh().setSolverPerformance(psi.name(), solverPerf);
*/
    SolverPerformance<Type> solverPerf; 
    return solverPerf;
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::solve
(
    const dictionary& solverControls
)
{
    return psi_.mesh().solve(*this, solverControls);
}


template<class Type>
Foam::autoPtr<typename Foam::gpufvMatrix<Type>::fvSolver>
Foam::gpufvMatrix<Type>::solver()
{
    return solver(solverDict());
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::fvSolver::solve()
{
    return solve(fvMat_.solverDict());
}


template<class Type>
Foam::SolverPerformance<Type> Foam::gpufvMatrix<Type>::solve()
{
    return this->solve(solverDict());
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::gpufvMatrix<Type>::residual() const
{
    tmp<gpuField<Type>> tres(new gpuField<Type>(source_));
    gpuField<Type>& res = tres.ref();

    addBoundarySource(res);

    // Loop over field components
    for (direction cmpt=0; cmpt<Type::nComponents; cmpt++)
    {
        scalargpuField psiCmpt(psi_.primitiveField().component(cmpt));

        scalargpuField boundaryDiagCmpt(psi_.size(), Zero);
        addBoundaryDiag(boundaryDiagCmpt, cmpt);

        FieldField<gpuField, scalar> bouCoeffsCmpt
        (
            boundaryCoeffs_.component(cmpt)
        );

        res.replace
        (
            cmpt,
            gpulduMatrix::residual
            (
                psiCmpt,
                res.component(cmpt) - boundaryDiagCmpt*psiCmpt,
                bouCoeffsCmpt,
                psi_.boundaryField().scalarInterfaces(),
                cmpt
            )
        );
    }

    return tres;
}


// ************************************************************************* //
