/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "coupledFvPatchgpuField.H"
#include "gpulduAddressingFunctors.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(p)),
    fvPatchgpuField<Type>(p, iF)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const Field<Type>& f
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(p)),
    fvPatchgpuField<Type>(p, iF, f)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const gpuField<Type>& f
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(p)),
    fvPatchgpuField<Type>(p, iF, f)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const coupledFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(p)),
    fvPatchgpuField<Type>(ptf, p, iF, mapper)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict,
    const bool valueRequired
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(p, dict)),
    fvPatchgpuField<Type>(p, iF, dict, valueRequired)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const coupledFvPatchgpuField<Type>& ptf
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(ptf.patch())),
    fvPatchgpuField<Type>(ptf)
{}


template<class Type>
Foam::coupledFvPatchgpuField<Type>::coupledFvPatchgpuField
(
    const coupledFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    LduInterfacegpuField<Type>(refCast<const gpulduInterface>(ptf.patch())),
    fvPatchgpuField<Type>(ptf, iF)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::coupledFvPatchgpuField<Type>::snGrad
(
    const scalargpuField& deltaCoeffs
) const
{
    return
        deltaCoeffs
       *(this->patchNeighbourField() - this->patchInternalField());
}


template<class Type>
void Foam::coupledFvPatchgpuField<Type>::initEvaluate(const Pstream::commsTypes)
{
    if (!this->updated())
    {
        this->updateCoeffs();
    }
}


template<class Type>
void Foam::coupledFvPatchgpuField<Type>::evaluate(const Pstream::commsTypes)
{
    if (!this->updated())
    {
        this->updateCoeffs();
    }

    gpuField<Type>::operator=
    (
        this->patch().weights()*this->patchInternalField()
      + (1.0 - this->patch().weights())*this->patchNeighbourField()
    );

    fvPatchgpuField<Type>::evaluate();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::valueInternalCoeffs
(
    const tmp<scalargpuField>& w
) const
{
    return Type(pTraits<Type>::one)*w;
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::valueBoundaryCoeffs
(
    const tmp<scalargpuField>& w
) const
{
    return Type(pTraits<Type>::one)*(1.0 - w);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::gradientInternalCoeffs
(
    const scalargpuField& deltaCoeffs
) const
{
    return -Type(pTraits<Type>::one)*deltaCoeffs;
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::gradientInternalCoeffs() const
{
    NotImplemented;
    return -Type(pTraits<Type>::one)*this->patch().deltaCoeffs();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::gradientBoundaryCoeffs
(
    const scalargpuField& deltaCoeffs
) const
{
    return -this->gradientInternalCoeffs(deltaCoeffs);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::coupledFvPatchgpuField<Type>::gradientBoundaryCoeffs() const
{
    NotImplemented;
    return -this->gradientInternalCoeffs();
}


template<class Type>
void Foam::coupledFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    this->writeEntry("value", os);
}


template<class Type>
void Foam::coupledFvPatchgpuField<Type>::updateInterfaceMatrix
(
    scalargpuField& result,
    const scalargpuField& coeffs,
    const scalargpuField& pnf,
    const bool add
) const
{
    if(add)
    {
        matrixPatchOperation
        (
            this->patch().index(),
            result,
            this->patch().boundaryMesh().mesh().lduAddr(),
            matrixInterfaceFunctor<scalar,true>
            (   
                coeffs.data(),
                pnf.data()
            )
        );
    }
    else
    {
        matrixPatchOperation
        (
            this->patch().index(),
            result,
            this->patch().boundaryMesh().mesh().lduAddr(),
            matrixInterfaceFunctor<scalar,false>
            (
                coeffs.data(),
                pnf.data()
            )
        );
    }
}


// ************************************************************************* //
