/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::laplacianScheme

Group
    grpFvLaplacianSchemes

Description
    Abstract base class for laplacian schemes.

SourceFiles
    laplacianScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpulaplacianScheme_H
#define gpulaplacianScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "gpulinear.H"
#include "gpucorrectedSnGrad.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

template<class Type>
class gpufvMatrix;

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class laplacianScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type, class GType>
class gpulaplacianScheme
:
    public refCount
{

protected:

    // Protected data

        const gpufvMesh& mesh_;
        tmp<gpusurfaceInterpolationScheme<GType>> tinterpGammaScheme_;
        tmp<gpusnGradScheme<Type>> tsnGradScheme_;


private:

    // Private Member Functions

        //- No copy construct
        gpulaplacianScheme(const gpulaplacianScheme&) = delete;

        //- No copy assignment
        void operator=(const gpulaplacianScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpulaplacianScheme,
            Istream,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpulaplacianScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh),
            tinterpGammaScheme_(new gpulinear<GType>(mesh)),
            tsnGradScheme_(new gpucorrectedSnGrad<Type>(mesh))
        {}

        //- Construct from mesh and Istream
        gpulaplacianScheme(const gpufvMesh& mesh, Istream& is)
        :
            mesh_(mesh),
            tinterpGammaScheme_(nullptr),
            tsnGradScheme_(nullptr)
        {
            tinterpGammaScheme_ = tmp<gpusurfaceInterpolationScheme<GType>>
            (
                gpusurfaceInterpolationScheme<GType>::New(mesh, is)
            );

            tsnGradScheme_ = tmp<gpusnGradScheme<Type>>
            (
                gpusnGradScheme<Type>::New(mesh, is)
            );
        }

        //- Construct from mesh, interpolation and snGradScheme schemes
        gpulaplacianScheme
        (
            const gpufvMesh& mesh,
            const tmp<gpusurfaceInterpolationScheme<GType>>& igs,
            const tmp<gpusnGradScheme<Type>>& sngs
        )
        :
            mesh_(mesh),
            tinterpGammaScheme_(igs),
            tsnGradScheme_(sngs)
        {}


    // Selectors

        //- Return a pointer to a new laplacianScheme created on freestore
        static tmp<gpulaplacianScheme<Type, GType>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpulaplacianScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        virtual tmp<gpufvMatrix<Type>> fvmLaplacian
        (
            const GeometricgpuField<GType, fvsPatchgpuField, gpusurfaceMesh>&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmLaplacian
        (
            const GeometricgpuField<GType, fvPatchgpuField, gpuvolMesh>&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcLaplacian
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcLaplacian
        (
            const GeometricgpuField<GType, fvsPatchgpuField, gpusurfaceMesh>&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcLaplacian
        (
            const GeometricgpuField<GType, fvPatchgpuField, gpuvolMesh>&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuLaplacianTypeScheme(SS, GType, Type)                             \
    typedef Foam::fv::SS<Foam::Type, Foam::GType> SS##Type##GType;             \
    defineNamedTemplateTypeNameAndDebug(SS##Type##GType, 0);                   \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            typedef SS<Type, GType> SS##Type##GType;                           \
                                                                               \
            gpulaplacianScheme<Type, GType>::                                     \
                addIstreamConstructorToTable<SS<Type, GType>>                  \
                add##SS##Type##GType##IstreamConstructorToTable_;              \
        }                                                                      \
    }


#define makeFvgpuLaplacianScheme(SS)                                              \
                                                                               \
makeFvgpuLaplacianTypeScheme(SS, scalar, scalar)                                  \
makeFvgpuLaplacianTypeScheme(SS, symmTensor, scalar)                              \
makeFvgpuLaplacianTypeScheme(SS, tensor, scalar)                                  \
makeFvgpuLaplacianTypeScheme(SS, scalar, vector)                                  \
makeFvgpuLaplacianTypeScheme(SS, symmTensor, vector)                              \
makeFvgpuLaplacianTypeScheme(SS, tensor, vector)                                  \
makeFvgpuLaplacianTypeScheme(SS, scalar, sphericalTensor)                         \
makeFvgpuLaplacianTypeScheme(SS, symmTensor, sphericalTensor)                     \
makeFvgpuLaplacianTypeScheme(SS, tensor, sphericalTensor)                         \
makeFvgpuLaplacianTypeScheme(SS, scalar, symmTensor)                              \
makeFvgpuLaplacianTypeScheme(SS, symmTensor, symmTensor)                          \
makeFvgpuLaplacianTypeScheme(SS, tensor, symmTensor)                              \
makeFvgpuLaplacianTypeScheme(SS, scalar, tensor)                                  \
makeFvgpuLaplacianTypeScheme(SS, symmTensor, tensor)                              \
makeFvgpuLaplacianTypeScheme(SS, tensor, tensor)

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpulaplacianScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
