/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2018 OpenFOAM Foundation
    Copyright (C) 2017 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::ddtScheme

Group
    grpFvDdtSchemes

Description
    Abstract base class for ddt schemes.

SourceFiles
    ddtScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuddtScheme_H
#define gpuddtScheme_H

#include "tmp.H"
#include "dimensionedType.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

template<class Type>
class gpufvMatrix;

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class ddtScheme Declaration
\*---------------------------------------------------------------------------*/

class gpuddtSchemeBase
{
public:

        //- Flag to use experimental ddtCorr from org version
        //- Default is off for backwards compatibility
        static bool experimentalDdtCorr;

        gpuddtSchemeBase()
        {}
};


template<class Type>
class gpuddtScheme
:
    public refCount,
    public gpuddtSchemeBase
{

protected:

    // Protected data

        const gpufvMesh& mesh_;

        //- Input for fvcDdtPhiCoeff
        //  If set to -1 (default) the code will calculate the coefficient
        //  If > 0 the coupling coeff is set to this value
        scalar ddtPhiCoeff_;


    // Private Member Functions

        //- No copy construct
        gpuddtScheme(const gpuddtScheme&) = delete;

        //- No copy assignment
        void operator=(const gpuddtScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpuddtScheme,
            Istream,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpuddtScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh),
            ddtPhiCoeff_(-1)
        {}

        //- Construct from mesh and Istream
        gpuddtScheme(const gpufvMesh& mesh, Istream& is)
        :
            mesh_(mesh),
            ddtPhiCoeff_(-1)
        {}


    // Selectors

        //- Return a pointer to a new ddtScheme created on freestore
        static tmp<gpuddtScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpuddtScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensioned<Type>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>&
        );

        virtual tmp<gpufvMatrix<Type>> fvmDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) = 0;

        typedef GeometricgpuField
        <
            typename flux<Type>::type,
            fvsPatchgpuField,
            gpusurfaceMesh
        > fluxFieldType;

        tmp<surfaceScalargpuField> fvcDdtPhiCoeff
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi,
            const fluxFieldType& phiCorr
        );

        tmp<surfaceScalargpuField> fvcDdtPhiCoeffExperimental
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi,
            const fluxFieldType& phiCorr
        );

        tmp<surfaceScalargpuField> fvcDdtPhiCoeff
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi,
            const fluxFieldType& phiCorr,
            const volScalargpuField& rho
        );

        tmp<surfaceScalargpuField> fvcDdtPhiCoeff
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<surfaceScalargpuField> fvcDdtPhiCoeff
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& rhoU,
            const fluxFieldType& phi,
            const volScalargpuField& rho
        );

        virtual tmp<fluxFieldType> fvcDdtUfCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        ) = 0;

        virtual tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        ) = 0;

        virtual tmp<fluxFieldType> fvcDdtUfCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        ) = 0;

        virtual tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        ) = 0;

        virtual tmp<surfaceScalargpuField> meshPhi
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuDdtTypeScheme(SS, Type)                                          \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpuddtScheme<Type>::addIstreamConstructorToTable<SS<Type>>            \
                add##SS##Type##IstreamConstructorToTable_;                     \
        }                                                                      \
    }

#define makeFvgpuDdtScheme(SS)                                                    \
                                                                               \
makeFvgpuDdtTypeScheme(SS, scalar)                                                \
makeFvgpuDdtTypeScheme(SS, vector)                                                \
makeFvgpuDdtTypeScheme(SS, sphericalTensor)                                       \
makeFvgpuDdtTypeScheme(SS, symmTensor)                                            \
makeFvgpuDdtTypeScheme(SS, tensor)                                                \
                                                                               \
namespace Foam                                                                 \
{                                                                              \
namespace fv                                                                   \
{                                                                              \
                                                                               \
template<>                                                                     \
tmp<surfaceScalargpuField> SS<scalar>::fvcDdtUfCorr                               \
(                                                                              \
    const volScalargpuField& U,                                                   \
    const surfaceScalargpuField& Uf                                               \
)                                                                              \
{                                                                              \
    NotImplemented;                                                            \
    return surfaceScalargpuField::null();                                         \
}                                                                              \
                                                                               \
template<>                                                                     \
tmp<surfaceScalargpuField> SS<scalar>::fvcDdtPhiCorr                              \
(                                                                              \
    const volScalargpuField& U,                                                   \
    const surfaceScalargpuField& phi                                              \
)                                                                              \
{                                                                              \
    NotImplemented;                                                            \
    return surfaceScalargpuField::null();                                         \
}                                                                              \
                                                                               \
template<>                                                                     \
tmp<surfaceScalargpuField> SS<scalar>::fvcDdtUfCorr                               \
(                                                                              \
    const volScalargpuField& rho,                                                 \
    const volScalargpuField& U,                                                   \
    const surfaceScalargpuField& Uf                                               \
)                                                                              \
{                                                                              \
    NotImplemented;                                                            \
    return surfaceScalargpuField::null();                                         \
}                                                                              \
                                                                               \
template<>                                                                     \
tmp<surfaceScalargpuField> SS<scalar>::fvcDdtPhiCorr                              \
(                                                                              \
    const volScalargpuField& rho,                                                 \
    const volScalargpuField& U,                                                   \
    const surfaceScalargpuField& phi                                              \
)                                                                              \
{                                                                              \
    NotImplemented;                                                            \
    return surfaceScalargpuField::null();                                         \
}                                                                              \
                                                                               \
}                                                                              \
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuddtScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
