/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::boundedDdtScheme

Group
    grpFvDdtSchemes

Description
    Bounded form of the selected ddt scheme.

    Boundedness is achieved by subtracting ddt(phi)*vf or Sp(ddt(rho), vf)
    which is non-conservative if ddt(rho) != 0 but conservative otherwise.

    Can be used for the ddt of bounded scalar properties to improve stability
    if insufficient convergence of the pressure equation causes temporary
    divergence of the flux field.

SourceFiles
    boundedDdtScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuboundedDdtScheme_H
#define gpuboundedDdtScheme_H

#include "gpuddtScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                       Class boundedDdtScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuboundedDdtScheme
:
    public fv::gpuddtScheme<Type>
{
    // Private data

        tmp<fv::gpuddtScheme<Type>> scheme_;


    // Private Member Functions

        //- No copy construct
        gpuboundedDdtScheme(const gpuboundedDdtScheme&) = delete;

        //- No copy assignment
        void operator=(const gpuboundedDdtScheme&) = delete;


public:

    //- Runtime type information
    TypeName("bounded");


    // Constructors

        //- Construct from mesh and Istream
        gpuboundedDdtScheme(const gpufvMesh& mesh, Istream& is)
        :
            gpuddtScheme<Type>(mesh, is),
            scheme_
            (
                fv::gpuddtScheme<Type>::New(mesh, is)
            )
        {}


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return fv::gpuddtScheme<Type>::mesh();
        }

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensioned<Type>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        /*tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>&
        );*/

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        typedef typename gpuddtScheme<Type>::fluxFieldType fluxFieldType;

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<surfaceScalargpuField> meshPhi
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );
};


template<>
tmp<surfaceScalargpuField> gpuboundedDdtScheme<scalar>::fvcDdtUfCorr
(
    const GeometricgpuField<scalar, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh>& Uf
);

template<>
tmp<surfaceScalargpuField> gpuboundedDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);

template<>
tmp<surfaceScalargpuField> gpuboundedDdtScheme<scalar>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& Uf
);

template<>
tmp<surfaceScalargpuField> gpuboundedDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuboundedDdtScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
