/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::PBiCGStab

Description
    Preconditioned bi-conjugate gradient stabilized solver for asymmetric
    lduMatrices using a run-time selectable preconditioner.

    References:
    \verbatim
        Van der Vorst, H. A. (1992).
        Bi-CGSTAB: A fast and smoothly converging variant of Bi-CG
        for the solution of nonsymmetric linear systems.
        SIAM Journal on scientific and Statistical Computing, 13(2), 631-644.

        Barrett, R., Berry, M. W., Chan, T. F., Demmel, J., Donato, J.,
        Dongarra, J., Eijkhout, V., Pozo, R., Romine, C. & Van der Vorst, H.
        (1994).
        Templates for the solution of linear systems:
        building blocks for iterative methods
        (Vol. 43). Siam.
    \endverbatim

SourceFiles
    PBiCGStab.C

\*---------------------------------------------------------------------------*/

#ifndef GPBiCGStab_H
#define GPBiCGStab_H

#include "gpulduMatrix.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class PBiCGStab Declaration
\*---------------------------------------------------------------------------*/

class GPBiCGStab
:
    public gpulduMatrix::solver
{
    // Private Member Functions

        //- No copy construct
        GPBiCGStab(const GPBiCGStab&) = delete;

        //- No copy assignment
        void operator=(const GPBiCGStab&) = delete;


public:

    //- Runtime type information
    TypeName("GPBiCGStab");


    // Constructors

        //- Construct from matrix components and solver data stream
        GPBiCGStab
        (
            const word& fieldName,
            const gpulduMatrix& matrix,
            const FieldField<gpuField, scalar>& interfaceBouCoeffs,
            const FieldField<gpuField, scalar>& interfaceIntCoeffs,
            const lduInterfacegpuFieldPtrsList& interfaces,
            const dictionary& solverControls
        );


    //- Destructor
    virtual ~GPBiCGStab() = default;


    // Member Functions

        //- Solve the matrix with this solver
        virtual solverPerformance scalarSolve
        (
            scalargpuField& psi,
            const scalargpuField& source,
            const direction cmpt = 0
        ) const;

        //- Solve the matrix with this solver
        virtual solverPerformance solve
        (
            scalargpuField& psi,
            const scalargpuField& source,
            const direction cmpt=0
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
