/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::GAMGInterfaceField

Description
     Abstract base class for GAMG agglomerated interface fields.

SourceFiles
    GAMGInterfaceField.C
    GAMGInterfaceFieldNew.C

\*---------------------------------------------------------------------------*/

#ifndef GGAMGInterfaceField_H
#define GGAMGInterfaceField_H

#include "lduInterfacegpuField.H"
#include "GGAMGInterface.H"
#include "autoPtr.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class GAMGInterfaceField Declaration
\*---------------------------------------------------------------------------*/

class GGAMGInterfaceField
:
    public lduInterfacegpuField
{
    // Private data

        //- Local reference cast into the interface
        const GGAMGInterface& interface_;


    // Private Member Functions

        //- No copy construct
        GGAMGInterfaceField(const GGAMGInterfaceField&) = delete;

        //- No copy assignment
        void operator=(const GGAMGInterfaceField&) = delete;


public:

    //- Runtime type information
    TypeName("GGAMGInterfaceField");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            autoPtr,
            GGAMGInterfaceField,
            lduInterfacegpuField,
            (
                const GGAMGInterface& GAMGCp,
                const lduInterfacegpuField& fineInterface
            ),
            (GAMGCp, fineInterface)
        );

        declareRunTimeSelectionTable
        (
            autoPtr,
            GGAMGInterfaceField,
            gpulduInterface,
            (
                const GGAMGInterface& GAMGCp,
                const bool doTransform,
                const int rank
            ),
            (GAMGCp, doTransform, rank)
        );


    // Selectors

        //- Return a pointer to a new interface created on freestore given
        //  the fine interface
        static autoPtr<GGAMGInterfaceField> New
        (
            const GGAMGInterface& GAMGCp,
            const lduInterfacegpuField& fineInterface
        );

        //- Return a pointer to a new interface created on freestore given
        //  the fine interface
        static autoPtr<GGAMGInterfaceField> New
        (
            const GGAMGInterface& GAMGCp,
            const bool doTransform,
            const int rank
        );


    // Constructors

        //- Construct from GAMG interface and fine level interface field
        GGAMGInterfaceField
        (
            const GGAMGInterface& GAMGCp,
            const lduInterfacegpuField&
        )
        :
            lduInterfacegpuField(GAMGCp),
            interface_(GAMGCp)
        {}

        //- Construct from GAMG interface and fine level interface field
        GGAMGInterfaceField
        (
            const GGAMGInterface& GAMGCp,
            const bool doTransform,
            const int rank
        )
        :
            lduInterfacegpuField(GAMGCp),
            interface_(GAMGCp)
        {}


    // Member Functions

        // Access

            //- Return interface
            const GGAMGInterface& interface() const
            {
                return interface_;
            }

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
