/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::GAMGSolver

Group
    grpLduMatrixSolvers

Description
    Geometric agglomerated algebraic multigrid solver.

    Characteristics:
      - Requires positive definite, diagonally dominant matrix.
      - Agglomeration algorithm: selectable and optionally cached.
      - Restriction operator: summation.
      - Prolongation operator: injection.
      - Smoother: Gauss-Seidel.
      - Coarse matrix creation: central coefficient: summation of fine grid
        central coefficients with the removal of intra-cluster face;
        off-diagonal coefficient: summation of off-diagonal faces.
      - Coarse matrix scaling: performed by correction scaling, using steepest
        descent optimisation.
      - Type of cycle: V-cycle with optional pre-smoothing.
      - Coarsest-level matrix solved using PCG or PBiCGStab.

SourceFiles
    GAMGSolver.C
    GAMGSolverAgglomerateMatrix.C
    GAMGSolverInterpolate.C
    GAMGSolverScale.C
    GAMGSolverSolve.C

\*---------------------------------------------------------------------------*/

#ifndef GGAMGSolver_H
#define GGAMGSolver_H

#include "GGAMGAgglomeration.H"
#include "gpulduMatrix.H"
#include "labelField.H"
#include "primitiveFields.H"
#include "gpuLUscalarMatrix.H"

#include "PageLockedBuffer.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                         Class GAMGSolver Declaration
\*---------------------------------------------------------------------------*/

class GGAMGSolver
:
    public gpulduMatrix::solver
{
    // Private data

        bool cacheAgglomeration_;

        //- Number of pre-smoothing sweeps
        label nPreSweeps_;

        //- Lever multiplier for the number of pre-smoothing sweeps
        label preSweepsLevelMultiplier_;

        //- Maximum number of pre-smoothing sweeps
        label maxPreSweeps_;

        //- Number of post-smoothing sweeps
        label nPostSweeps_;

        //- Lever multiplier for the number of post-smoothing sweeps
        label postSweepsLevelMultiplier_;

        //- Maximum number of post-smoothing sweeps
        label maxPostSweeps_;

        //- Number of smoothing sweeps on finest mesh
        label nFinestSweeps_;

        //- Choose if the corrections should be interpolated after injection.
        //  By default corrections are not interpolated.
        bool interpolateCorrection_;

        //- Choose if the corrections should be scaled.
        //  By default corrections for symmetric matrices are scaled
        //  but not for asymmetric matrices.
        bool scaleCorrection_;

        //- Direct or iteratively solve the coarsest level
        bool directSolveCoarsest_;

        //- The agglomeration
        const GGAMGAgglomeration& agglomeration_;

        //- Hierarchy of matrix levels
        PtrList<gpulduMatrix> matrixLevels_;

        //- Hierarchy of interfaces.
        PtrList<PtrList<lduInterfacegpuField>> primitiveInterfaceLevels_;

        //- Hierarchy of interfaces in lduInterfaceFieldPtrs form
        PtrList<lduInterfacegpuFieldPtrsList> interfaceLevels_;

        //- Hierarchy of interface boundary coefficients
        PtrList<FieldField<gpuField, scalar>> interfaceLevelsBouCoeffs_;

        //- Hierarchy of interface internal coefficients
        PtrList<FieldField<gpuField, scalar>> interfaceLevelsIntCoeffs_;

        //- LU decomposed coarsest matrix
        autoPtr<gpuLUscalarMatrix> coarsestLUMatrixPtr_;

        //- Sparse coarsest matrix solver
        autoPtr<gpulduMatrix::solver> coarsestSolverPtr_;

        //- Coarsest matrix result buffer
        mutable scalarField* coarsestBufferPtr_;


    // Private Member Functions

        //- Read control parameters from the control dictionary
        virtual void readControls();

        //- Simplified access to interface level
        const lduInterfacegpuFieldPtrsList& interfaceLevel
        (
            const label i
        ) const;

        //- Simplified access to matrix level
        const gpulduMatrix& matrixLevel(const label i) const;

        //- Simplified access to interface boundary coeffs level
        const FieldField<gpuField, scalar>& interfaceBouCoeffsLevel
        (
            const label i
        ) const;

        //- Simplified access to interface internal coeffs level
        const FieldField<gpuField, scalar>& interfaceIntCoeffsLevel
        (
            const label i
        ) const;

        //- Agglomerate coarse matrix. Supply mesh to use - so we can
        //  construct temporary matrix on the fine mesh (instead of the coarse
        //  mesh)
        void agglomerateMatrix
        (
            const label fineLevelIndex,
            const gpulduMesh& coarseMesh,
            const gpulduInterfacePtrsList& coarseMeshInterfaces
        );

        //- Agglomerate coarse interface coefficients
        void agglomerateInterfaceCoefficients
        (
            const label fineLevelIndex,
            const gpulduInterfacePtrsList& coarseMeshInterfaces,
            PtrList<lduInterfacegpuField>& coarsePrimInterfaces,
            lduInterfacegpuFieldPtrsList& coarseInterfaces,
            FieldField<gpuField, scalar>& coarseInterfaceBouCoeffs,
            FieldField<gpuField, scalar>& coarseInterfaceIntCoeffs
        ) const;

        //- Collect matrices from other processors
        /*
        void gatherMatrices
        (
            const labelList& procIDs,
            const lduMesh& dummyMesh,
            const label meshComm,

            const lduMatrix& mat,
            const FieldField<Field, scalar>& interfaceBouCoeffs,
            const FieldField<Field, scalar>& interfaceIntCoeffs,
            const lduInterfaceFieldPtrsList& interfaces,

            PtrList<lduMatrix>& otherMats,
            PtrList<FieldField<Field, scalar>>& otherBouCoeffs,
            PtrList<FieldField<Field, scalar>>& otherIntCoeffs,
            List<boolList>& otherTransforms,
            List<List<label>>& otherRanks
        ) const;

        //- Agglomerate processor matrices
        void procAgglomerateMatrix
        (
            // Agglomeration information
            const labelList& procAgglomMap,
            const List<label>& agglomProcIDs,

            const label levelI,

            // Resulting matrix
            autoPtr<lduMatrix>& allMatrixPtr,
            FieldField<Field, scalar>& allInterfaceBouCoeffs,
            FieldField<Field, scalar>& allInterfaceIntCoeffs,
            PtrList<lduInterfaceField>& allPrimitiveInterfaces,
            lduInterfaceFieldPtrsList& allInterfaces
        ) const;

        //- Agglomerate processor matrices
        void procAgglomerateMatrix
        (
            const labelList& procAgglomMap,
            const List<label>& agglomProcIDs,
            const label levelI
        );
        */

        //- Interpolate the correction after injected prolongation
        void interpolate
        (
            scalargpuField& psi,
            scalargpuField& Apsi,
            const gpulduMatrix& m,
            const FieldField<gpuField, scalar>& interfaceBouCoeffs,
            const lduInterfacegpuFieldPtrsList& interfaces,
            const direction cmpt
        ) const;

        //- Interpolate the correction after injected prolongation and
        //  re-normalise
        void interpolate
        (
            scalargpuField& psi,
            scalargpuField& Apsi,
            const gpulduMatrix& m,
            const FieldField<gpuField, scalar>& interfaceBouCoeffs,
            const lduInterfacegpuFieldPtrsList& interfaces,
            const labelgpuList& restrictSortAddressing,
            const labelgpuList& restrictTargetAddressing,
            const labelgpuList& restrictTargetStartAddressing,
            const scalargpuField& psiC,
            const direction cmpt
        ) const;

        //- Calculate and apply the scaling factor from Acf, coarseSource
        //  and coarseField.
        //  At the same time do a Jacobi iteration on the coarseField using
        //  the Acf provided after the coarseField values are used for the
        //  scaling factor.
        void scale
        (
            scalargpuField& field,
            scalargpuField& Acf,
            const gpulduMatrix& A,
            const FieldField<gpuField, scalar>& interfaceLevelBouCoeffs,
            const lduInterfacegpuFieldPtrsList& interfaceLevel,
            const scalargpuField& source,
            const direction cmpt
        ) const;

        //- Initialise the data structures for the V-cycle
        void initVcycle
        (
            PtrList<scalargpuField>& coarseCorrFields,
            PtrList<scalargpuField>& coarseSources,
            PtrList<gpulduMatrix::smoother>& smoothers,
            scalargpuField& scratch1,
            scalargpuField& scratch2
        ) const;


        //- Perform a single GAMG V-cycle with pre, post and finest smoothing.
        void Vcycle
        (
            const PtrList<gpulduMatrix::smoother>& smoothers,
            scalargpuField& psi,
            const scalargpuField& source,
            scalargpuField& Apsi,
            scalargpuField& finestCorrection,
            scalargpuField& finestResidual,

            scalargpuField& scratch1,
            scalargpuField& scratch2,

            PtrList<scalargpuField>& coarseCorrFields,
            PtrList<scalargpuField>& coarseSources,
            const direction cmpt=0
        ) const;

        //- Create and return the dictionary to specify the PCG solver
        //  to solve the coarsest level
        dictionary GPCGsolverDict
        (
            const scalar tol,
            const scalar relTol
        ) const;

        //- Create and return the dictionary to specify the PBiCGStab solver
        //  to solve the coarsest level
        dictionary GPBiCGStabSolverDict
        (
            const scalar tol,
            const scalar relTol
        ) const;

        //- Solve the coarsest level with either an iterative or direct solver
        void solveCoarsestLevel
        (
            scalargpuField& coarsestCorrField,
            const scalargpuField& coarsestSource
        ) const;
      
        static PageLockedBuffer<scalar> coarseBuffer;

public:

    friend class GAMGPreconditioner;

    //- Runtime type information
    TypeName("GAMG");


    // Constructors

        //- Construct from lduMatrix and solver controls
        GGAMGSolver
        (
            const word& fieldName,
            const gpulduMatrix& matrix,
            const FieldField<gpuField, scalar>& interfaceBouCoeffs,
            const FieldField<gpuField, scalar>& interfaceIntCoeffs,
            const lduInterfacegpuFieldPtrsList& interfaces,
            const dictionary& solverControls
        );


    //- Destructor
    virtual ~GGAMGSolver();


    // Member Functions

        //- Solve
        virtual solverPerformance solve
        (
            scalargpuField& psi,
            const scalargpuField& source,
            const direction cmpt=0
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
