/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2015-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "heThermo.H"
#include "gradientEnergyFvPatchScalargpuField.H"
#include "mixedEnergyFvPatchScalargpuField.H"

namespace Foam
{
    template<class Mixture>
	struct heThermoHEFunctor
	{		
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.HE(p,T);
		}
	};

	template<class Mixture>
	struct heThermoHcFunctor
	{
		__host__ __device__
		scalar operator () (const Mixture& mixture)
		{
			return mixture.Hc();
		}
	};
	
	template<class Mixture>
	struct heThermoCpFunctor
	{		
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.Cp(p,T);
		}
	};	

	
	template<class Mixture>
	struct heThermoCvFunctor{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.Cv(p,T);
		}
	};

	template<class Mixture>
	struct heThermoRhoFunctor
	{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.rho(p,T);
		}
	};
	
    template<class Mixture>
	struct heThermoGammaFunctor
	{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.gamma(p,T);
		}
	};
	
	template<class Mixture>
	struct heThermoCpvFunctor
	{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.Cpv(p,T);
		}
	};
	
	template<class Mixture>
	struct heThermoCpByCpvFunctor
	{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar p = thrust::get<0>(t);
			const scalar T = thrust::get<1>(t);
			return mixture.CpByCpv(p,T);
		}
	};
	
	template<class Mixture>
	struct heThermoTHEFunctor
	{
		__host__ __device__
		scalar operator () (const thrust::tuple<scalar,scalar,scalar>& t, const Mixture& mixture)
		{
			const scalar h = thrust::get<0>(t);
			const scalar p = thrust::get<1>(t);
			const scalar T = thrust::get<2>(t);
			return mixture.THE(h,p,T);
		}
	};


	template<class Mixture>
	struct heThermoWFunctor
	{
		__host__ __device__
		scalar operator () (const Mixture& mixture)
		{
			return mixture.W();
		}
	};

}

// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //

template<class BasicThermo, class MixtureType>
void Foam::heThermo<BasicThermo, MixtureType>::
heBoundaryCorrection(volScalargpuField& h)
{
    volScalargpuField::Boundary& hBf = h.boundaryFieldRef();

    forAll(hBf, patchi)
    {
        if (isA<gradientEnergyFvPatchScalargpuField>(hBf[patchi]))
        {
            refCast<gradientEnergyFvPatchScalargpuField>(hBf[patchi]).gradient()
                = hBf[patchi].fvPatchgpuField::snGrad();
        }
        else if (isA<mixedEnergyFvPatchScalargpuField>(hBf[patchi]))
        {
            refCast<mixedEnergyFvPatchScalargpuField>(hBf[patchi]).refGrad()
                = hBf[patchi].fvPatchgpuField::snGrad();
        }
    }
}


template<class BasicThermo, class MixtureType>
void Foam::heThermo<BasicThermo, MixtureType>::init
(
    const volScalargpuField& p,
    const volScalargpuField& T,
    volScalargpuField& he
)
{
    scalargpuField& heCells = he.primitiveFieldRef();
    const scalargpuField& pCells = p.primitiveField();
    const scalargpuField& TCells = T.primitiveField();

    this->updateCellMixture();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(pCells.begin(), TCells.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(pCells.end(), TCells.end())),
				this->mixtureCells(), 
				heCells.begin(),
                heThermoHEFunctor<typename MixtureType::thermoType>()
				);


    volScalargpuField::Boundary& heBf = he.boundaryFieldRef();

    forAll(heBf, patchi)
    {
        heBf[patchi] == this->he
        (
            p.boundaryField()[patchi],
            T.boundaryField()[patchi],
            patchi
        );

        heBf[patchi].useImplicit(T.boundaryField()[patchi].useImplicit());
    }

    this->heBoundaryCorrection(he);

    // Note: T does not have oldTime
    if (p.nOldTimes() > 0)
    {
        init(p.oldTime(), T.oldTime(), he.oldTime());
    }
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class BasicThermo, class MixtureType>
Foam::heThermo<BasicThermo, MixtureType>::heThermo
(
    const gpufvMesh& mesh,
    const word& phaseName
)
:
    BasicThermo(mesh, phaseName),
    MixtureType(*this, mesh, phaseName),

    he_
    (
        IOobject
        (
            BasicThermo::phasePropertyName
            (
                MixtureType::thermoType::heName()
            ),
            mesh.time().timeName(),
            mesh.hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE
        ),
        mesh,
        dimEnergy/dimMass,
        this->heBoundaryTypes(),
        this->heBoundaryBaseTypes()
    )
{
    init(this->p_, this->T_, he_);
}


template<class BasicThermo, class MixtureType>
Foam::heThermo<BasicThermo, MixtureType>::heThermo
(
    const gpufvMesh& mesh,
    const dictionary& dict,
    const word& phaseName
)
:
    BasicThermo(mesh, dict, phaseName),
    MixtureType(*this, mesh, phaseName),

    he_
    (
        IOobject
        (
            BasicThermo::phasePropertyName
            (
                MixtureType::thermoType::heName()
            ),
            mesh.time().timeName(),
            mesh.hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE
        ),
        mesh,
        dimEnergy/dimMass,
        this->heBoundaryTypes(),
        this->heBoundaryBaseTypes()
    )
{
    init(this->p_, this->T_, he_);
}


template<class BasicThermo, class MixtureType>
Foam::heThermo<BasicThermo, MixtureType>::heThermo
(
    const gpufvMesh& mesh,
    const word& phaseName,
    const word& dictionaryName
)
:
    BasicThermo(mesh, phaseName, dictionaryName),
    MixtureType(*this, mesh, phaseName),

    he_
    (
        IOobject
        (
            BasicThermo::phasePropertyName
            (
                MixtureType::thermoType::heName()
            ),
            mesh.time().timeName(),
            mesh.hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE
        ),
        mesh,
        dimEnergy/dimMass,
        this->heBoundaryTypes(),
        this->heBoundaryBaseTypes()
    )
{
    init(this->p_, this->T_, he_);
}



// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

template<class BasicThermo, class MixtureType>
Foam::heThermo<BasicThermo, MixtureType>::~heThermo()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField> Foam::heThermo<BasicThermo, MixtureType>::he
(
    const volScalargpuField& p,
    const volScalargpuField& T
) const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> the
    (
        new volScalargpuField
        (
            IOobject
            (
                "he",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            he_.dimensions()
        )
    );

    volScalargpuField& he = the.ref();
    scalargpuField& heCells = he.primitiveFieldRef();
    const scalargpuField& pCells = p.primitiveField();
    const scalargpuField& TCells = T.primitiveField();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(pCells.begin(), TCells.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(pCells.end(), TCells.end())),
				this->mixtureCells(), 
				heCells.begin(),
                heThermoHEFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& heBf = he.boundaryFieldRef();

    forAll(heBf, patchi)
    {
        scalargpuField& hep = heBf[patchi];
        const scalargpuField& pp = p.boundaryField()[patchi];
        const scalargpuField& Tp = T.boundaryField()[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);
        
        thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(pp.begin(), Tp.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(pp.end(), Tp.end())),
				mixtureFacesPtr, 
				hep.begin(),
                heThermoHEFunctor< typename MixtureType::thermoType >()
				);

    }

    return the;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::he
(
    const scalargpuField& p,
    const scalargpuField& T,
    const labelgpuList& cells
) const
{
    tmp<scalargpuField> the(new scalargpuField(T.size()));
    scalargpuField& he = the.ref();

	thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				he.begin(),
                heThermoHEFunctor< typename MixtureType::thermoType >()
				);

    return the;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::he
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> the(new scalargpuField(T.size()));
    scalargpuField& he = the.ref();
	
    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				mixtureFacesPtr,
				he.begin(),
                heThermoHEFunctor< typename MixtureType::thermoType >()
				);

    return the;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::hc() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> thc
    (
        new volScalargpuField
        (
            IOobject
            (
                "hc",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            he_.dimensions()
        )
    );

    volScalargpuField& hcf = thc.ref();
    scalargpuField& hcCells = hcf.primitiveFieldRef();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureCellsPtr = this->mixtureCells();

	thrust::transform(
				mixtureCellsPtr,
                mixtureCellsPtr + hcCells.size(),
				hcCells.begin(),
                heThermoHcFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& hcfBf = hcf.boundaryFieldRef();

    forAll(hcfBf, patchi)
    {
        scalargpuField& hcp = hcfBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

		thrust::transform(
				mixtureFacesPtr,
                mixtureFacesPtr + hcp.size(),
				hcp.begin(),
                heThermoHcFunctor< typename MixtureType::thermoType >()
				);
    }
    return thc;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::Cp
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> tCp(new scalargpuField(T.size()));
    scalargpuField& cp = tCp.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				mixtureFacesPtr, 
				cp.begin(),
                heThermoCpFunctor< typename MixtureType::thermoType >()
				);

    return tCp;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::Cp
(
    const scalargpuField& p,
    const scalargpuField& T,
    const labelgpuList& cells
) const
{
    auto tCp = tmp<scalargpuField>::New(T.size());
    auto& Cp = tCp.ref();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				Cp.begin(),
                heThermoCpFunctor< typename MixtureType::thermoType >()
				);

    return tCp;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::Cp() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tCp
    (
        new volScalargpuField
        (
            IOobject
            (
                "Cp",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimEnergy/dimMass/dimTemperature
        )
    );

    volScalargpuField& cp = tCp.ref();
	const scalargpuField& p=this->p_.primitiveField();
	const scalargpuField& T=this->T_.primitiveField();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				cp.primitiveFieldRef().begin(),
                heThermoCpFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& cpBf = cp.boundaryFieldRef();

    forAll(cpBf, patchi)
    {
        const fvPatchScalargpuField& pp = this->p_.boundaryField()[patchi];
        const fvPatchScalargpuField& pT = this->T_.boundaryField()[patchi];
        fvPatchScalargpuField& pCp = cpBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

        thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(pp.begin(), pT.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(pp.end(), pT.end())),
				mixtureFacesPtr, 
				pCp.begin(),
                heThermoCpFunctor< typename MixtureType::thermoType >()
				);

    }

    return tCp;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::Cv
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> tCv(new scalargpuField(T.size()));
    scalargpuField& cv = tCv.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

    thrust::transform(
			thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
            thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
			mixtureFacesPtr, 
			cv.begin(),
            heThermoCvFunctor< typename MixtureType::thermoType >()
			);

    return tCv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::rhoEoS
(
    const scalargpuField& p,
    const scalargpuField& T,
    const labelgpuList& cells
) const
{
    auto tRho = tmp<scalargpuField>::New(T.size());
    auto& rho = tRho.ref();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				rho.begin(),
                heThermoRhoFunctor< typename MixtureType::thermoType >()
				);

    return tRho;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::Cv() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tCv
    (
        new volScalargpuField
        (
            IOobject
            (
                "Cv",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimEnergy/dimMass/dimTemperature
        )
    );

    volScalargpuField& cv = tCv.ref();
	const scalargpuField& p = this->p_.primitiveField();
	const scalargpuField& T = this->T_.primitiveField();
    scalargpuField& pcv = cv.primitiveFieldRef();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				pcv.begin(),
                heThermoCvFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& cvBf = cv.boundaryFieldRef();

    forAll(cvBf, patchi)
    {
        cvBf[patchi] = Cv
        (
            this->p_.boundaryField()[patchi],
            this->T_.boundaryField()[patchi],
            patchi
        );
    }

    return tCv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::gamma
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> tgamma(new scalargpuField(T.size()));
    scalargpuField& gamma = tgamma.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

    thrust::transform(
	        thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
            thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
			mixtureFacesPtr, 
		    gamma.begin(),
            heThermoGammaFunctor< typename MixtureType::thermoType >()
			);

    return tgamma;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::gamma() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tgamma
    (
        new volScalargpuField
        (
            IOobject
            (
                "gamma",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimless
        )
    );

    volScalargpuField& gamma = tgamma.ref();
	const scalargpuField& p=this->p_.primitiveField();
	const scalargpuField& T=this->T_.primitiveField();
    scalargpuField& ggamma = gamma.primitiveFieldRef();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				ggamma.begin(),
                heThermoGammaFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& gammaBf = gamma.boundaryFieldRef();

    forAll(gammaBf, patchi)
    {
        const fvPatchScalargpuField& pp = this->p_.boundaryField()[patchi];
        const fvPatchScalargpuField& pT = this->T_.boundaryField()[patchi];
        fvPatchScalargpuField& pgamma = gammaBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

		thrust::transform(
	        thrust::make_zip_iterator(thrust::make_tuple(pp.begin(), pT.begin())),
            thrust::make_zip_iterator(thrust::make_tuple(pp.end(), pT.end())),
			mixtureFacesPtr, 
		    pgamma.begin(),
            heThermoGammaFunctor< typename MixtureType::thermoType >()
			);
    }

    return tgamma;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::Cpv
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> tCpv(new scalargpuField(T.size()));
    scalargpuField& Cpv = tCpv.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

	thrust::transform(
	    thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
        thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
		mixtureFacesPtr, 
		Cpv.begin(),
        heThermoCpvFunctor< typename MixtureType::thermoType >()
		);

    return tCpv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::Cpv() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tCpv
    (
        new volScalargpuField
        (
            IOobject
            (
                "Cpv",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimEnergy/dimMass/dimTemperature
        )
    );

    volScalargpuField& Cpv = tCpv.ref();
	const scalargpuField& p=this->p_.primitiveField();
	const scalargpuField& T=this->T_.primitiveField();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				Cpv.primitiveFieldRef().begin(),
                heThermoCpvFunctor< typename MixtureType::thermoType >()
				);


    volScalargpuField::Boundary& CpvBf = Cpv.boundaryFieldRef();

    forAll(CpvBf, patchi)
    {
        const fvPatchScalargpuField& pp = this->p_.boundaryField()[patchi];
        const fvPatchScalargpuField& pT = this->T_.boundaryField()[patchi];
        fvPatchScalargpuField& pCpv = CpvBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);
 
		thrust::transform(
	        thrust::make_zip_iterator(thrust::make_tuple(pp.begin(), pT.begin())),
            thrust::make_zip_iterator(thrust::make_tuple(pp.end(), pT.end())),
			mixtureFacesPtr, 
		    pCpv.begin(),
            heThermoCpvFunctor< typename MixtureType::thermoType >()
			);
    }

    return tCpv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::CpByCpv
(
    const scalargpuField& p,
    const scalargpuField& T,
    const label patchi
) const
{
    tmp<scalargpuField> tCpByCpv(new scalargpuField(T.size()));
    scalargpuField& CpByCpv = tCpByCpv.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

	thrust::transform(
	    thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
        thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
		mixtureFacesPtr, 
		CpByCpv.begin(),
        heThermoCpByCpvFunctor< typename MixtureType::thermoType >()
		);

    return tCpByCpv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::CpByCpv() const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tCpByCpv
    (
        new volScalargpuField
        (
            IOobject
            (
                "CpByCpv",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimless
        )
    );

    volScalargpuField& CpByCpv = tCpByCpv.ref();
	const scalargpuField& p=this->p_.primitiveField();
	const scalargpuField& T=this->T_.primitiveField();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(p.begin(), T.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(p.end(), T.end())),
				this->mixtureCells(), 
				CpByCpv.primitiveFieldRef().begin(),
                heThermoCpByCpvFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& CpByCpvBf =
        CpByCpv.boundaryFieldRef();

    forAll(CpByCpvBf, patchi)
    {
        const fvPatchScalargpuField& pp = this->p_.boundaryField()[patchi];
        const fvPatchScalargpuField& pT = this->T_.boundaryField()[patchi];
        fvPatchScalargpuField& pCpByCpv = CpByCpvBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

		thrust::transform(
	        thrust::make_zip_iterator(thrust::make_tuple(pp.begin(), pT.begin())),
            thrust::make_zip_iterator(thrust::make_tuple(pp.end(), pT.end())),
			mixtureFacesPtr, 
		    pCpByCpv.begin(),
            heThermoCpByCpvFunctor< typename MixtureType::thermoType >()
			);
    }

    return tCpByCpv;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::THE
(
    const scalargpuField& h,
    const scalargpuField& p,
    const scalargpuField& T0,
    const labelgpuList& cells
) const
{
    tmp<scalargpuField> tT(new scalargpuField(h.size()));
    scalargpuField& T = tT.ref();

    thrust::transform(
				thrust::make_zip_iterator(thrust::make_tuple(h.begin(), p.begin(), T0.begin())),
                thrust::make_zip_iterator(thrust::make_tuple(h.end(), p.end(), T0.end())),
				this->mixtureCells(), 
				T.begin(),
                heThermoTHEFunctor< typename MixtureType::thermoType >()
				);

    return tT;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::THE
(
    const scalargpuField& h,
    const scalargpuField& p,
    const scalargpuField& T0,
    const label patchi
) const
{

    tmp<scalargpuField> tT(new scalargpuField(h.size()));
    scalargpuField& T = tT.ref();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);

	thrust::transform(
	    thrust::make_zip_iterator(thrust::make_tuple(h.begin(), p.begin(), T0.begin())),
        thrust::make_zip_iterator(thrust::make_tuple(h.end(), p.end(), T0.end())),
		mixtureFacesPtr, 
		T.begin(),
        heThermoTHEFunctor< typename MixtureType::thermoType >()
		);

    return tT;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField> Foam::heThermo<BasicThermo, MixtureType>::W
(
) const
{
    const gpufvMesh& mesh = this->T_.mesh();

    tmp<volScalargpuField> tW
    (
        new volScalargpuField
        (
            IOobject
            (
                "W",
                mesh.time().timeName(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh,
            dimMass/dimMoles
        )
    );

    volScalargpuField& W = tW.ref();
    scalargpuField& WCells = W.primitiveFieldRef();

    const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureCellsPtr = this->mixtureCells();

    thrust::transform(
				mixtureCellsPtr,
                mixtureCellsPtr + WCells.size(),
				WCells.begin(),
                heThermoWFunctor< typename MixtureType::thermoType >()
				);

    volScalargpuField::Boundary& WBf = W.boundaryFieldRef();

    forAll(WBf, patchi)
    {
        scalargpuField& Wp = WBf[patchi];

        const thrust::device_ptr<const typename MixtureType::thermoType>& mixtureFacesPtr = this->updatePatchFaceMixture(patchi);		

		thrust::transform( 
			mixtureFacesPtr,
            mixtureFacesPtr + Wp.size(),
		    Wp.begin(),
            heThermoWFunctor< typename MixtureType::thermoType >()
			);
    }

    return tW;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::kappa() const
{
    tmp<Foam::volScalargpuField> kappa(Cp()*this->alpha_);
    kappa.ref().rename("kappa");
    return kappa;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField> Foam::heThermo<BasicThermo, MixtureType>::kappa
(
    const label patchi
) const
{
    return
        Cp
        (
            this->p_.boundaryField()[patchi],
            this->T_.boundaryField()[patchi],
            patchi
        )*this->alpha_.boundaryField()[patchi];
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::alphahe() const
{
    tmp<Foam::volScalargpuField> alphaEff(this->CpByCpv()*this->alpha_);
    alphaEff.ref().rename("alphahe");
    return alphaEff;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::alphahe(const label patchi) const
{
    return
    this->CpByCpv
    (
        this->p_.boundaryField()[patchi],
        this->T_.boundaryField()[patchi],
        patchi
    )
   *this->alpha_.boundaryField()[patchi];
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::kappaEff
(
    const volScalargpuField& alphat
) const
{
    tmp<Foam::volScalargpuField> kappaEff(Cp()*(this->alpha_ + alphat));
    kappaEff.ref().rename("kappaEff");
    return kappaEff;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::kappaEff
(
    const scalargpuField& alphat,
    const label patchi
) const
{
    return
        Cp
        (
            this->p_.boundaryField()[patchi],
            this->T_.boundaryField()[patchi],
            patchi
        )
       *(
           this->alpha_.boundaryField()[patchi]
         + alphat
        );
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::volScalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::alphaEff
(
    const volScalargpuField& alphat
) const
{
    tmp<Foam::volScalargpuField> alphaEff(this->CpByCpv()*(this->alpha_ + alphat));
    alphaEff.ref().rename("alphaEff");
    return alphaEff;
}


template<class BasicThermo, class MixtureType>
Foam::tmp<Foam::scalargpuField>
Foam::heThermo<BasicThermo, MixtureType>::alphaEff
(
    const scalargpuField& alphat,
    const label patchi
) const
{
    return
    this->CpByCpv
    (
        this->p_.boundaryField()[patchi],
        this->T_.boundaryField()[patchi],
        patchi
    )
   *(
        this->alpha_.boundaryField()[patchi]
      + alphat
    );
}


template<class BasicThermo, class MixtureType>
bool Foam::heThermo<BasicThermo, MixtureType>::read()
{
    if (BasicThermo::read())
    {
        MixtureType::read(*this);
        return true;
    }

    return false;
}


// ************************************************************************* //
