/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2012 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "mixedEnergyFvPatchScalargpuField.H"
#include "addToRunTimeSelectionTable.H"
#include "fvPatchgpuFieldMapper.H"
#include "volgpuFields.H"
#include "basicThermo.H"

#include "gpufvMatrix.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::mixedEnergyFvPatchScalargpuField::
mixedEnergyFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    mixedFvPatchScalargpuField(p, iF)
{
    valueFraction() = 0.0;
    refValue() = 0.0;
    refGrad() = 0.0;
    source() = 0.0;
}


Foam::mixedEnergyFvPatchScalargpuField::
mixedEnergyFvPatchScalargpuField
(
    const mixedEnergyFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    mixedFvPatchScalargpuField(ptf, p, iF, mapper)
{}


Foam::mixedEnergyFvPatchScalargpuField::
mixedEnergyFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    mixedFvPatchScalargpuField(p, iF, dict)
{}


Foam::mixedEnergyFvPatchScalargpuField::
mixedEnergyFvPatchScalargpuField
(
    const mixedEnergyFvPatchScalargpuField& tppsf
)
:
    mixedFvPatchScalargpuField(tppsf)
{}


Foam::mixedEnergyFvPatchScalargpuField::
mixedEnergyFvPatchScalargpuField
(
    const mixedEnergyFvPatchScalargpuField& tppsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    mixedFvPatchScalargpuField(tppsf, iF)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::mixedEnergyFvPatchScalargpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }
    const basicThermo& thermo = basicThermo::lookupThermo(*this);
    const label patchi = patch().index();

    const scalargpuField& pw = thermo.p().boundaryField()[patchi];
    mixedFvPatchScalargpuField& Tw = refCast<mixedFvPatchScalargpuField>
    (
        const_cast<fvPatchScalargpuField&>(thermo.T().boundaryField()[patchi])
    );

    Tw.evaluate();

    valueFraction() = Tw.valueFraction();
    refValue() = thermo.he(pw, Tw.refValue(), patchi);
    refGrad() =
        thermo.Cpv(pw, Tw, patchi)*Tw.refGrad()
      + patch().deltaCoeffs()*
        (
            thermo.he(pw, Tw, patchi)
          - thermo.he(pw, Tw, patch().gpuFaceCells())
        );

    mixedFvPatchScalargpuField::updateCoeffs();
}


void Foam::mixedEnergyFvPatchScalargpuField::manipulateMatrix
(
    gpufvMatrix<scalar>& matrix,
    const label mat,
    const direction cmpt
)
{
    const basicThermo& thermo = basicThermo::lookupThermo(*this);

    label index = this->patch().index();

    const label nbrPatchId =  this->patch().patch().neighbPolyPatchID();

    const label globalPatchID =
        matrix.lduMeshAssembly().patchLocalToGlobalMap()[mat][index];

    const label meshNrbId = matrix.lduMeshAssembly().findNbrMeshId
    (
        this->patch().patch(),
        mat
    );

    const mixedFvPatchgpuField<scalar>& fPatch =
        refCast<const mixedFvPatchgpuField>(thermo.T().boundaryField()[index]);

    const gpuField<scalar> intCoeffsCmpt
    (
        matrix.internalCoeffs()[globalPatchID].component(cmpt)
    );

    const scalargpuField sourceCorr(fPatch.source());

    const labelList& faceMap =
        matrix.lduMeshAssembly().faceBoundMap()[mat][index];

    const labelList& myCells =
        matrix.lduMeshAssembly().cellBoundMap()[meshNrbId][nbrPatchId];

    const labelList& nbrCells =
        matrix.lduMeshAssembly().cellBoundMap()[mat][index];

    forAll(faceMap, j)
    {
        label globalFaceI = faceMap[j];

        label myCellI = myCells[j];
        label nbrCellI = nbrCells[j];

        const scalar intCorr = -intCoeffsCmpt.get(j);
        const scalar srcCorr = -sourceCorr.get(j);

        if (this->patch().patch().masterImplicit())
        {
            if (myCellI > nbrCellI)
            {
                if (matrix.asymmetric())
                {
                    //matrix.lower()[globalFaceI] += intCorr;
                    const scalar vl =  matrix.gpuLower().get(globalFaceI);
                    matrix.gpuLower().set(globalFaceI, vl + intCorr);
                }
            }
            else
            {
                //matrix.upper()[globalFaceI] += intCorr;
                const scalar vu =  matrix.gpuUpper().get(globalFaceI);
                matrix.gpuUpper().set(globalFaceI, vu + intCorr);
            }

            const scalar vd = matrix.gpuDiag().get(myCellI);
            const scalar vs = matrix.source().get(myCellI);
            matrix.gpuDiag().set(myCellI, vd + intCorr);
            matrix.source().set(myCellI, vs + srcCorr);
            //matrix.diag()[myCellI] -= intCorr;
            //matrix.source()[myCellI] += srcCorr;
        }
        else
        {
            if (myCellI < nbrCellI)
            {
                //matrix.upper()[globalFaceI] += intCorr;
                const scalar vu =  matrix.gpuUpper().get(globalFaceI);
                matrix.gpuUpper().set(globalFaceI, vu + intCorr);
            }
            else
            {
                if (matrix.asymmetric())
                {
                    //matrix.lower()[globalFaceI] += intCorr;
                    const scalar vl =  matrix.gpuLower().get(globalFaceI);
                    matrix.gpuLower().set(globalFaceI, vl + intCorr);
                }
            }

            //matrix.diag()[myCellI] -= intCorr;
            //matrix.source()[myCellI] += srcCorr;
            const scalar vd = matrix.gpuDiag().get(myCellI);
            const scalar vs = matrix.source().get(myCellI);
            matrix.gpuDiag().set(myCellI, vd + intCorr);
            matrix.source().set(myCellI, vs + srcCorr);
        }


//         if (globalFaceI != -1)
//         {
//             const scalar intCorr = -intCoeffsCmpt[j];
//             const scalar srcCorr = -sourceCorr[j];
//
//             if (this->patch().patch().masterImplicit())
//             {
//                 matrix.diag()[u[globalFaceI]] -= intCorr;
//                 if (matrix.asymmetric())
//                 {
//                     matrix.lower()[globalFaceI] += intCorr;
//                 }
//                 matrix.source()[u[globalFaceI]] += srcCorr;
//             }
//             else
//             {
//                 matrix.diag()[l[globalFaceI]] -= intCorr;
//                 matrix.upper()[globalFaceI] += intCorr;
//                 matrix.source()[l[globalFaceI]] += srcCorr;
//             }
//         }
    }
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        mixedEnergyFvPatchScalargpuField
    );
}

// ************************************************************************* //
