/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::uniform

Description
    Particle-size distribution model wherein random samples are drawn
    from the doubly-truncated uniform probability density function:

    \f[
        f(x; A, B) = \frac{1}{B - A}
    \f]
    where

    \vartable
      f(x; A, B)  | Doubly-truncated uniform distribution
      x           | Sample
      A           | Minimum of the distribution
      B           | Maximum of the distribution
    \endvartable

    Constraints:
    - \f$ \infty > B > A > 0 \f$
    - \f$ x \in [B,A] \f$
    - \f$ \sigma^2 > 0 \f$

    Random samples are generated by the inverse transform sampling technique
    by using the quantile function of the uniform probability density function:

    \f[
        x = u \, (B - A) + A
    \f]

    where \f$ u \f$ is sample drawn from the uniform probability
    density function on the unit interval \f$ (0, 1) \f$.

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...

                sizeDistribution
                {
                    type        uniform;
                    uniformDistribution
                    {
                        minValue  <min>;
                        maxValue  <max>;
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property   | Description                       | Type | Reqd   | Deflt
      type       | Type name: uniform                | word | yes    | -
      uniformDistribution | Distribution settings    | dict | yes    | -
      minValue   | Minimum of the distribution       | scalar | yes  | -
      maxValue   | Maximum of the distribution       | scalar | yes  | -
    \endtable

SourceFiles
    uniform.C

\*---------------------------------------------------------------------------*/

#ifndef distributionModels_uniform_H
#define distributionModels_uniform_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                           Class uniform Declaration
\*---------------------------------------------------------------------------*/

class uniform
:
    public distributionModel
{
public:

    //- Runtime type information
    TypeName("uniform");


    // Constructors

        //- Construct from components
        uniform(const dictionary& dict, Random& rndGen);

        //- Copy construct
        uniform(const uniform& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new uniform(*this));
        }

        //- No copy assignment
        void operator=(const uniform&) = delete;


    //- Destructor
    virtual ~uniform() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
